package com.bxm.newidea.component.redisson.impl;

import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import com.bxm.newidea.component.redisson.config.RedissonConfigurationProperties;
import com.bxm.newidea.component.redisson.config.SwitchMultiDataSourceHandler;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;

import java.util.concurrent.TimeUnit;

/**
 * 基于Redisson实现的分布式锁
 * 这里只是一种简单的实现，更加复杂的锁实现可以参见：
 * <p>
 * 官方文档：<br/>
 * https://github.com/redisson/redisson/wiki/8.-%E5%88%86%E5%B8%83%E5%BC%8F%E9%94%81%E5%92%8C%E5%90%8C%E6%AD%A5%E5%99%A8
 * <p>
 * <p>
 * 源作者思路：<br/>
 * https://zhuanlan.zhihu.com/p/38501022
 * </p>
 *
 * @author liujia 2018/8/13 21:04
 */
@Slf4j
public class RedissonDistributedLockImpl implements DistributedLock {

    private SwitchMultiDataSourceHandler switchMultiDataSourceHandler;

    private RedissonConfigurationProperties redissonConfigurationProperties;

    public RedissonDistributedLockImpl(SwitchMultiDataSourceHandler switchMultiDataSourceHandler,
                                       RedissonConfigurationProperties redissonConfigurationProperties) {
        this.switchMultiDataSourceHandler = switchMultiDataSourceHandler;
        this.redissonConfigurationProperties = redissonConfigurationProperties;
    }

    @Override
    public boolean lock(String resource) {
        return this.lock(resource, 5, TimeUnit.SECONDS);
    }

    @Override
    public boolean lock(String key, String requestId) {
        // 兼容接口，忽略requestId参数，redisson已经内置了防止错误解锁的机制
        return this.lock(key, 5, TimeUnit.SECONDS);
    }

    @Override
    public boolean lock(String resource, long time, TimeUnit timeUnit) {
        String key = buildKey(resource);
        RLock rlock = switchMultiDataSourceHandler.getRedissonClient(redissonConfigurationProperties.getDistributeDataSource()).getLock(key);

        try {
            return rlock.tryLock(-1, time, timeUnit);
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);

            rlock.unlock();

            Thread.currentThread().interrupt();

            return false;
        }
    }

    private String buildKey(String resource) {
        return DefaultKeyGenerator.build("biz", "db", resource).gen();
    }

    /**
     * 尝试获取锁
     *
     * @param lockKey
     * @param unit
     * @param waitTime
     * @param leaseTime
     * @return
     */
    private boolean tryLock(String lockKey, TimeUnit unit, int waitTime, int leaseTime) {
        String key = buildKey(lockKey);
        RLock lock = switchMultiDataSourceHandler.getRedissonClient(redissonConfigurationProperties.getDistributeDataSource()).getLock(key);
        try {
            return lock.tryLock(waitTime, leaseTime, unit);
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);
            lock.unlock();
            Thread.currentThread().interrupt();
            return false;
        }
    }


    @Override
    public void unlock(String resource) {
        String key = buildKey(resource);
        RLock lock = switchMultiDataSourceHandler.getRedissonClient(redissonConfigurationProperties.getDistributeDataSource()).getLock(key);
        try {
            lock.unlock();
        } catch (Exception e) {
            log.warn("当前key锁资源不存在，解锁也没用，resource：{}", resource, e);
        }
    }

    @Override
    public void unlock(String key, String requestId) {
        // 兼容接口，忽略requestId参数，redisson已经内置了防止错误解锁的机制
        unlock(key);
    }

}
