package com.bxm.newidea.component.redisson.config;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.beans.factory.support.BeanDefinitionRegistry;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.ImportBeanDefinitionRegistrar;
import org.springframework.core.env.Environment;
import org.springframework.core.type.AnnotationMetadata;
import org.springframework.util.Assert;

import java.util.Map;

/**
 * 手工注入多个配置文件中定义的
 *
 * @author wzy
 * @version 1.0
 * @date 2021/8/5 2:46 下午
 */
@Slf4j
public class RedisClientRegister implements ImportBeanDefinitionRegistrar, EnvironmentAware {

    private Environment environment;

    /**
     * 参数绑定工具 springboot2.0新推出
     */
    private Binder binder;

    @Override
    public void registerBeanDefinitions(AnnotationMetadata importingClassMetadata,
                                        BeanDefinitionRegistry registry) {

        RedissonConfigurationProperties configurationMetaInfos =
                binder.bind("component.redisson", RedissonConfigurationProperties.class).get();

        //如果没有默认的配置直接让项目启动不起来
        Assert.notNull(configurationMetaInfos, "未配置Redisson默认数据源，请检查配置！");

        Map<String, RedissonMetaInfo> configs = configurationMetaInfos.getMultiData();

        //将默认的Redisson数据源放到集合中
        configs.put("default", buildDefaultRedissonMetaInfo(configurationMetaInfos));

        RedissonClientHolder redissonClientHolder = new RedissonClientHolder();

        RedissonClientBeanDefinitionFactory redissonClientBeanDefinitionFactory = new RedissonClientBeanDefinitionFactory(redissonClientHolder);

        String[] activeProfiles = environment.getActiveProfiles();

        //循环遍历配置初始化Redisson数据源
        configs.forEach((dataSourceName, metaInfo) -> {
                    BeanDefinition beanDefinition = redissonClientBeanDefinitionFactory.get(dataSourceName,
                            metaInfo,
                            activeProfiles);

                    registry.registerBeanDefinition(dataSourceName + "RedissonDataSource", beanDefinition);
                }
        );
    }

    /**
     * 构建默认redisson数据源信息
     *
     * @param configurationMetaInfos redisson数据源配置
     * @return redisson数据源信息
     */
    private RedissonMetaInfo buildDefaultRedissonMetaInfo(RedissonConfigurationProperties configurationMetaInfos) {
        RedissonMetaInfo redissonMetaInfo = new RedissonMetaInfo();

        BeanUtils.copyProperties(configurationMetaInfos, redissonMetaInfo);

        return redissonMetaInfo;
    }

    @Override
    public void setEnvironment(Environment environment) {
        this.environment = environment;
        log.info("开始注册数据源");
        // 绑定配置器
        binder = Binder.get(environment);
    }
}