package com.bxm.newidea.component.redisson.impl;

import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RLock;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;

/**
 * 基于Redisson实现的分布式锁
 * 这里只是一种简单的实现，更加复杂的锁实现可以参见：
 * <p>
 * 官方文档：<br/>
 * https://github.com/redisson/redisson/wiki/8.-%E5%88%86%E5%B8%83%E5%BC%8F%E9%94%81%E5%92%8C%E5%90%8C%E6%AD%A5%E5%99%A8
 * <p>
 * <p>
 * 源作者思路：<br/>
 * https://zhuanlan.zhihu.com/p/38501022
 * </p>
 *
 * @author liujia 2018/8/13 21:04
 */
@Component
@Slf4j
public class RedissonDistributedLockImpl implements DistributedLock {

    @Resource
    private RedissonClient redissonClient;

    @Autowired
    public RedissonDistributedLockImpl(RedissonClient redissonClient) {
        this.redissonClient = redissonClient;
    }

    @Override
    public boolean lock(String resource) {
        return this.lock(resource, 5, TimeUnit.SECONDS);
    }

    @Override
    public boolean lock(String resource, long time, TimeUnit timeUnit) {
        String key = buildKey(resource);
        RLock rlock = redissonClient.getLock(key);

        try {
            return rlock.tryLock(-1, time, timeUnit);
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);

            rlock.unlock();

            Thread.currentThread().interrupt();

            return false;
        }
    }

    private String buildKey(String resource) {
        return DefaultKeyGenerator.build("biz", "db", resource).gen();
    }

    /**
     * 尝试获取锁
     *
     * @param lockKey
     * @param unit
     * @param waitTime
     * @param leaseTime
     * @return
     */
    private boolean tryLock(String lockKey, TimeUnit unit, int waitTime, int leaseTime) {
        String key = buildKey(lockKey);
        RLock lock = redissonClient.getLock(key);
        try {
            return lock.tryLock(waitTime, leaseTime, unit);
        } catch (InterruptedException e) {
            log.error(e.getMessage(), e);
            lock.unlock();
            Thread.currentThread().interrupt();
            return false;
        }
    }


    @Override
    public void unlock(String resource) {
        String key = buildKey(resource);
        RLock lock = redissonClient.getLock(key);
        try {
            lock.unlock();
        } catch (Exception e) {
            log.warn("当前kay锁资源不存在，解锁也没用，resource：{}", resource, e);
        }
    }

}
