package com.bxm.newidea.component.redisson.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ArrayUtils;
import org.redisson.api.RMap;
import org.redisson.api.RedissonClient;
import org.redisson.client.codec.StringCodec;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.util.Assert;

import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.newidea.component.tools.NumberUtils.parseToLong;


@Component
@Slf4j
public class RedissonHashMapAdapterImpl extends BaseRedisAdapter implements RedisHashMapAdapter {


    @Autowired
    public RedissonHashMapAdapterImpl(RedissonClient redissonClient) {
        super(redissonClient);
    }

    private RMap hashOperations(KeyGenerator generator) {
        return redissonClient.getMap(generator.gen());
    }

    private RMap numericHashOperations(KeyGenerator generator) {
        return redissonClient.getMap(generator.gen(), StringCodec.INSTANCE);
    }

    /**
     * 把值添加到散列，并且关注此subKey是否存在，存在则返回原有value，并更新value为当前传递的值
     *
     * @param generator 散列键名
     * @param subKey    散列存储的键
     * @param value     散列存储的值
     * @param <T>
     */
    @Override
    public <T> void put(KeyGenerator generator, String subKey, T value) {
        Assert.notNull(generator, "key构建器必须传递");
        this.hashOperations(generator).put(subKey, this.serializerString(value));
    }

    /**
     * 把值添加到散列，不关注此subKey是否存在，只关注添加是否成功，
     * true则表明map中无此subKey，
     * false表明map中有此subKey，并更新value
     *
     * @param generator 散列键名
     * @param subKey    散列存储的键
     * @param value     散列存储的值
     * @param <T>
     */
    public <T> void fastPut(KeyGenerator generator, String subKey, T value) {
        Assert.notNull(generator, "key构建器必须传递");
        this.hashOperations(generator).fastPut(subKey, this.serializerString(value));
    }

    @Override
    public void putLong(KeyGenerator generator, String subKey, Long value) {
        if (generator == null || null == value) {
            return;
        }
        this.numericHashOperations(generator).put(subKey, value);
    }

    @Override
    public <T> void putAll(KeyGenerator generator, Map<String, T> values) {
        if (values == null) {
            return;
        }
        Map<String, String> convertResult = Maps.newHashMap();
        values.forEach((key, value) -> convertResult.put(key, this.serializerString(value)));
        this.hashOperations(generator).putAll(convertResult);
    }


    @Override
    public Long getLong(KeyGenerator generator, String subKey) {
        String value = (String) this.numericHashOperations(generator).get(subKey);
        if (null == value) {
            return 0L;
        }
        return parseToLong(value);
    }

    @Override
    public <T> T get(KeyGenerator generator, String subKey, Class<T> classes) {
        String value = (String) this.hashOperations(generator).get(subKey);
        if (null == value) {
            return null;
        }
        return this.deSerializerString(value, classes, null);
    }

    @Override
    public <T> T get(KeyGenerator generator, String subKey, TypeReference<T> typeReference) {
        String value = (String) this.hashOperations(generator).get(subKey);
        if (null == value) {
            return null;
        }

        return this.deSerializerString(value, null, typeReference);
    }

    @Override
    public Set<String> keys(KeyGenerator generator) {
        return this.hashOperations(generator).readAllKeySet();
    }

    @Override
    public <T> List<T> values(KeyGenerator generator, Class<T> clasz) {
        Collection<String> collection = this.hashOperations(generator).readAllValues();
        return (List<T>) collection.stream().map(item -> this.deSerializerString(item, clasz, null)).collect(Collectors.toList());
    }

    @Override
    public <T> List<T> values(KeyGenerator generator, TypeReference<T> typeReference) {
        Collection<String> collection = this.hashOperations(generator).readAllValues();
        return (List<T>) collection.stream().map(item -> this.deSerializerString(item, null, typeReference)).collect(Collectors.toList());
    }


    @Override
    public <T> List<T> multiGet(KeyGenerator generator, Collection<String> subKeys, Class<T> clasz) {
        return (List<T>) subKeys.stream().map(item -> {
            String value = (String) this.hashOperations(generator).get(item);
            if (null == value) {
                return null;
            }
            return this.deSerializerString(value, clasz, null);
        }).collect(Collectors.toList());
    }


    @Override
    public <T> List<T> multiGet(KeyGenerator generator, Collection<String> subKeys, TypeReference<T> typeReference) {
        return (List<T>) subKeys.stream().map(item -> {
            String value = (String) this.hashOperations(generator).get(item);
            if (null == value) {
                return null;
            }
            return this.deSerializerString(value, null, typeReference);
        }).collect(Collectors.toList());
    }

    @Override
    public <T> Map<String, T> entries(KeyGenerator generator, Class<T> clasz) {
        return getEntries(generator, clasz, null);
    }

    @Override
    public <T> Map<String, T> entries(KeyGenerator generator, TypeReference<T> typeReference) {
        return getEntries(generator, null, typeReference);
    }

    private <T> Map<String, T> getEntries(KeyGenerator generator, Class<T> clasz, TypeReference<T> typeReference) {
        Map<String, String> map = this.hashOperations(generator).readAllMap();
        Map<String, T> result = new HashMap<>();
        map.forEach((key, value) -> {
            result.put(key, this.deSerializerString(value, clasz, typeReference));

        });
        return result;
    }

    @Override
    public Boolean exists(KeyGenerator generator, String subKey) {
        return this.hashOperations(generator).containsKey(subKey);
    }


    @Override
    public Long remove(KeyGenerator generator, String... subKeys) {
        if (ArrayUtils.isEmpty(subKeys)) {
            return 0L;
        }
        return this.hashOperations(generator).fastRemove(subKeys);
    }

    @Override
    public Long length(KeyGenerator generator) {
        return (long) this.hashOperations(generator).size();
    }

    @Override
    public Long increment(KeyGenerator generator, String subKey, int amount) {
        Object value = this.numericHashOperations(generator).addAndGet(subKey, amount);
        if (null == value) {
            return 0L;
        }
        return Long.valueOf(value.toString());
    }

}
