package com.bxm.newidea.component.redisson.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.lang3.StringUtils;
import org.redisson.api.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.concurrent.TimeUnit;

@Component
public class RedissonStringAdapterImpl extends BaseRedisAdapter implements RedisStringAdapter {


    @Autowired
    public RedissonStringAdapterImpl(RedissonClient redissonClient) {
        super(redissonClient);
    }

    private RBucket<Object> stringOperations(KeyGenerator generator) {
        return redissonClient.getBucket(generator.gen());
    }

    private RTopic topOperations(String key) {
        return redissonClient.getTopic(key);
    }

    private RAtomicDouble doubleOperations(KeyGenerator generator) {
        return redissonClient.getAtomicDouble(generator.gen());
    }

    private RAtomicLong longOperations(KeyGenerator generator) {
        return redissonClient.getAtomicLong(generator.gen());
    }

    @Override
    public Long increment(KeyGenerator generator) {
        return this.increment(generator, 1);
    }

    @Override
    public Long increment(KeyGenerator generator, int amount) {
        return this.incrementWithDefault(generator, 1, amount, 0);
    }

    @Override
    public Long increment(KeyGenerator generator, long expireSeconds) {
        return this.incrementWithDefault(generator, 1, 1, expireSeconds);
    }

    @Override
    public Long incrementWithDefault(KeyGenerator generator, long defaultVal) {
        return this.incrementWithDefault(generator, defaultVal, 1, 0);
    }

    @Override
    public Long incrementWithDefault(KeyGenerator generator, long defaultVal, int amount) {
        return this.incrementWithDefault(generator, defaultVal, amount, 0);
    }

    @Override
    public Long incrementWithDefault(KeyGenerator generator, long defaultVal, int amount, long expireSeconds) {
        return this.exec(generator, defaultVal, amount, expireSeconds, true);
    }

    @Override
    public Long decrement(KeyGenerator generator, long defaultVal, int amount, long expireSeconds) {
        return this.exec(generator, defaultVal, amount, expireSeconds, false);
    }

    @Override
    public Long decrement(KeyGenerator generator) {
        return this.decrement(generator, 1, 1, 0);
    }

    @Override
    public Long decrement(KeyGenerator generator, int amount) {
        return this.decrement(generator, 1, amount, 0);
    }

    @Override
    public Long decrement(KeyGenerator generator, long expireSeconds) {
        return this.decrement(generator, 1, 1, expireSeconds);
    }

    @Override
    public Long decrement(KeyGenerator generator, long defaultVal, int amount) {
        return this.decrement(generator, defaultVal, amount, 0);
    }

    @Override
    public Long getLong(KeyGenerator generator) {
        return (long) this.doubleOperations(generator).get();
    }

    @Override
    public Double getDouble(KeyGenerator generator) {
        return this.doubleOperations(generator).get();
    }

    @Override
    public Integer getInt(KeyGenerator generator) {
        return (int) this.doubleOperations(generator).get();
    }

    @Override
    public String getString(KeyGenerator generator) {
        Object value = this.stringOperations(generator).get();
        if (null == value) {
            return StringUtils.EMPTY;
        }
        return String.valueOf(value);
    }

    @Override
    public <T> T get(KeyGenerator generator, Class<T> clasz) {
        Object value = this.stringOperations(generator).get();
        if (null == value) {
            return null;
        }
        return this.deSerializerString(String.valueOf(value), clasz, null);
    }

    @Override
    public <T> T get(KeyGenerator generator, TypeReference<T> typeReference) {
        Object value = this.stringOperations(generator).get();
        if (null == value) {
            return null;
        }
        return this.deSerializerString(String.valueOf(value), null, typeReference);
    }

    @Override
    public <T> void set(KeyGenerator generator, T value) {
        this.stringOperations(generator).set(this.serializerString(value));
    }

    @Override
    public void set(KeyGenerator generator, long value) {
        this.longOperations(generator).set(value);
    }

    @Override
    public <T> void set(KeyGenerator generator, T value, long expireSeconds) {
        if (value instanceof String) {
            this.stringOperations(generator).set(value.toString(), expireSeconds, TimeUnit.SECONDS);
        } else {
            this.stringOperations(generator).set(this.serializerString(value), expireSeconds, TimeUnit.SECONDS);
        }
    }

    @Override
    public void set(KeyGenerator generator, Double value) {
        this.doubleOperations(generator).set(value);
    }

    @Override
    public Double increment(KeyGenerator generator, double amount) {
        return this.increment(generator, amount, amount, 0);
    }

    @Override
    public Double increment(KeyGenerator generator, double defaultVal, double amount, long expireSeconds) {
        return this.exec(generator, defaultVal, amount, expireSeconds, true);
    }

    @Override
    public Double decrement(KeyGenerator generator, double amount) {
        return this.decrement(generator, 0, amount, 0);
    }

    @Override
    public Double decrement(KeyGenerator generator, double defaultVal, double amount, long expireSeconds) {
        return this.exec(generator, defaultVal, amount, expireSeconds, false);
    }


    @Override
    public void convertAndSend(String channel, Object message) {
        topOperations(channel).publish(message);
    }

    private Long exec(KeyGenerator generator, long defaultVal, long amount, long expireSeconds, boolean isIncrement) {
        if (!this.hasKey(generator)) {
            this.longOperations(generator).set(defaultVal);
            if (expireSeconds > 0) {
                this.longOperations(generator).expire(expireSeconds, TimeUnit.SECONDS);
            }
            return defaultVal;
        }
        amount = isIncrement ? amount : -amount;
        return this.longOperations(generator).addAndGet(amount);
    }

    private Double exec(KeyGenerator generator, double defaultVal, double amount, long expireSeconds, boolean isIncrement) {
        if (!this.hasKey(generator)) {
            this.doubleOperations(generator).set(defaultVal);
            if (expireSeconds > 0) {
                this.doubleOperations(generator).expire(expireSeconds, TimeUnit.SECONDS);
            }
            return defaultVal;
        }
        amount = isIncrement ? amount : -amount;
        return this.doubleOperations(generator).addAndGet(amount);
    }
}
