package com.bxm.newidea.component.redis;

import com.bxm.newidea.component.redis.impl.DefaultKeyGenerator;
import com.google.common.collect.Lists;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.core.script.DefaultRedisScript;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.concurrent.TimeUnit;

/**
 * 基于redis实现的分布式锁
 * FIXME [liujia]并不好用，可以考虑升级替换为redssion
 *
 * @author liujia 2018/8/13 21:04
 */
@Component
public class DistributedLock {

    @Resource
    private RedisTemplate redisTemplate;

    @Autowired
    public DistributedLock(RedisTemplate redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    /**
     * 锁定业务资源，防止并发情况对锁定的资源进行重复操作,默认锁定5秒。锁定结束后会自动解锁
     *
     * @param resource  锁定的资源
     * @param reqeustId 加锁的客户端ID，需保证唯一
     * @return true表示加锁成功，可进行后续业务，false表示加锁失败，需要进行适当的重试机制
     */
    public boolean lock(String resource, String reqeustId) {
        return this.lock(resource, reqeustId, 5, TimeUnit.SECONDS);
    }

    /**
     * 锁定业务资源，防止并发情况对锁定的资源进行重复操作
     *
     * @param resource  锁定的资源,如业务ID、编码等
     * @param reqeustId 请求方提供，需要保证每次请求不一样，防止分布式服务产生或使用相同的值，请求方需要持有该值，用于手动解锁
     * @param time      加锁的时长
     * @param timeUnit  加锁时间的单位
     * @return true表示加锁成功，可进行后续业务
     */
    public boolean lock(String resource, String reqeustId, long time, TimeUnit timeUnit) {
        String key = buildKey(resource);
        Boolean result = this.redisTemplate.opsForValue().setIfAbsent(key, reqeustId.getBytes());
        this.redisTemplate.expire(key, time, timeUnit);
        return result;
    }

    private String buildKey(String resource) {
        return DefaultKeyGenerator.build("biz", "db", resource).gen();
    }

    /**
     * 解锁资源
     *
     * @param resource  解锁的资源,如业务ID、编码等
     * @param requestId 加锁时传入的值，防止被其他业务错误解锁
     * @return true表示解锁成功，false表示解锁失败
     */
    public boolean unlock(String resource, String requestId) {
        String key = buildKey(resource);

        String script = "if redis.call('get', KEYS[1]) == ARGV[1] then return redis.call('del', KEYS[1]) else return 0 end";

        byte[] requestByteArray = requestId.getBytes();
        Object result = this.redisTemplate.execute(new DefaultRedisScript(script, Long.class), Lists.newArrayList(key), requestByteArray);
        return new Long(1L).equals(result);
    }

}
