package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.BaseRedisOperation;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.serializer.SupplementJackson2JsonRedisSerializer;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import org.apache.commons.lang3.ArrayUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.data.redis.serializer.StringRedisSerializer;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * 基础redis操作代理
 *
 * @author liujia 2018/5/21 17:35
 */
public class BaseRedisAdapter implements BaseRedisOperation {

    RedisTemplate<String, Object> redisTemplate;

    RedisSerializer<String> stringSerializer;

    private Logger logger = LoggerFactory.getLogger(this.getClass());

    public BaseRedisAdapter(RedisTemplate<String, Object> redisTemplate) {
        this.redisTemplate = redisTemplate;
        this.stringSerializer = new StringRedisSerializer();
    }

    @Override
    public Boolean remove(KeyGenerator key) {
        return this.redisTemplate.delete(key.gen());
    }

    @Override
    public Long remove(List<KeyGenerator> keys) {
        if (CollectionUtils.isEmpty(keys)) {
            return 0L;
        }

        List<String> finalKeys = keys.stream().map(KeyGenerator::gen).collect(Collectors.toList());
        return this.redisTemplate.delete(finalKeys);
    }

    @Override
    public void expire(KeyGenerator key, long seconds) {
        if (seconds > 0) {
            this.redisTemplate.expire(key.gen(), seconds, TimeUnit.SECONDS);
        }
    }

    @Override
    public void expire(KeyGenerator key, Date expiredDate) {
        if (null != expiredDate) {
            this.redisTemplate.expireAt(key.gen(), expiredDate);
        }
    }

    @Override
    public Boolean hasKey(KeyGenerator key) {
        return this.redisTemplate.hasKey(key.gen());
    }

    <T> RedisSerializer getSerializer(Class<T> clasz) {
        return new SupplementJackson2JsonRedisSerializer(clasz);
    }

    <T> RedisSerializer getSerializer(TypeReference<T> typeReference) {
        return new SupplementJackson2JsonRedisSerializer(typeReference);
    }

    RedisSerializer getSerializerWithoutType() {
        return new SupplementJackson2JsonRedisSerializer(Object.class);
    }

    <T> T deserialize(Object value, Class<T> clasz) {
        if (value instanceof byte[]) {
            return (T) this.getSerializer(clasz).deserialize((byte[]) value);
        }
        if (null == value) {
            return null;
        }
        this.logger.warn("redis返回数据不能进行反序列化，请检查是否配置了自定义的序列");
        return (T) value;
    }

    <T> T deserialize(Object value, TypeReference<T> typeReference) {
        if (value instanceof byte[]) {
            return (T) this.getSerializer(typeReference).deserialize((byte[]) value);
        }
        if (null == value) {
            return null;
        }
        this.logger.warn("redis返回数据不能进行反序列化，请检查是否配置了自定义的序列");
        return (T) value;
    }

    protected byte[][] serialize(String[] values) {
        byte[][] result = new byte[values.length][];

        for (int i = 0; i < values.length; i++) {
            result[i] = this.stringSerializer.serialize(values[i]);
        }

        return result;
    }

    <T> byte[][] serialize(Object[] values) {
        byte[][] result = new byte[values.length][];

        RedisSerializer serializer = getSerializerWithoutType();
        for (int i = 0; i < values.length; i++) {
            result[i] = serializer.serialize(values[i]);
        }

        return result;
    }

    <T> byte[][] serializeWithoutType(T[] values) {
        return serialize(values);
    }

    List<String> convertKeys(KeyGenerator... generators) {
        if (ArrayUtils.isEmpty(generators)) {
            return Lists.newArrayList();
        }

        List<String> keys = new ArrayList<>();

        for (KeyGenerator generator : generators) {
            keys.add(generator.gen());
        }

        return keys;
    }

    @Override
    public void rename(KeyGenerator oldKey, KeyGenerator newKey) {

        if (Objects.isNull(oldKey) || Objects.isNull(newKey)) {
            return;
        }
        redisTemplate.rename(oldKey.gen(), newKey.gen());
    }

}
