package com.bxm.newidea.component.redis.impl;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import org.apache.commons.lang3.ArrayUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.redis.core.ListOperations;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.data.redis.serializer.RedisSerializer;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.TimeUnit;

@SuppressWarnings("unchecked")
@Component
public class RedisListAdapterImpl extends BaseRedisAdapter implements RedisListAdapter {

    private ListOperations operations;

    @Autowired
    public RedisListAdapterImpl(RedisTemplate<String, Object> redisTemplate) {
        super(redisTemplate);
        operations = redisTemplate.opsForList();
    }

    @Override
    public Long leftPush(KeyGenerator generator, Object... values) {
        return this.execPush(generator, true, values);
    }

    @Override
    public <T> Long leftPush(KeyGenerator generator, List<T> value) {
        if (!CollectionUtils.isEmpty(value)) {
            return this.execPush(generator, true, value.toArray());
        }
        return 0L;
    }

    private <T> Long execPush(KeyGenerator generator, boolean isLeft, T[] values) {
        if (ArrayUtils.isEmpty(values)) {
            return 0L;
        }

        String key = generator.gen();

        try {
            List<byte[]> result = this.convertToByteArray(values);

            redisTemplate.watch(key);
            redisTemplate.multi();

            return isLeft ? operations.leftPushAll(key, result) : operations.rightPushAll(key, result);

        } finally {
            redisTemplate.unwatch();
        }
    }

    private <T> List<byte[]> convertToByteArray(T... values) {
        List<byte[]> dataArray = new ArrayList<>();

        for (T obj : values) {
            dataArray.add(getSerializerWithoutType().serialize(obj));
        }

        return dataArray;
    }

    @Override
    public <T> Long rightPush(KeyGenerator generator, T... values) {
        return this.execPush(generator, false, values);
    }

    @Override
    public Long size(KeyGenerator generator) {
        return operations.size(generator.gen());
    }

    @Override
    public <T> T leftPop(KeyGenerator generator, Class<T> clasz) {
        return this.execPop(generator, true, clasz, null);
    }

    @Override
    public <T> T blockLeftPop(KeyGenerator generator, Class<T> clasz, long timeout, TimeUnit unit) {
        return blockExecPop(generator, true, clasz, null, timeout, unit);
    }

    private <T> T blockExecPop(KeyGenerator generator,
                               boolean isLeft,
                               Class<T> clasz,
                               TypeReference<T> typeReference,
                               long timeout,
                               TimeUnit unit) {
        try {
            String key = generator.gen();

            redisTemplate.watch(key);
            redisTemplate.multi();

            Object value = isLeft ?
                    operations.leftPop(generator.gen(), timeout, unit) :
                    operations.rightPop(generator.gen(), timeout, unit);

            return this.deserialize(value, clasz);
        } finally {
            redisTemplate.unwatch();
        }
    }

    @Override
    public <T> T leftPop(KeyGenerator generator, TypeReference<T> typeReference) {
        return this.execPop(generator, true, null, typeReference);
    }

    private <T> T execPop(KeyGenerator generator, boolean isLeft, Class<T> clasz, TypeReference<T> typeReference) {
        try {
            String key = generator.gen();

            redisTemplate.watch(key);
            redisTemplate.multi();

            Object value = isLeft ? operations.leftPop(generator.gen()) : operations.rightPop(generator.gen());

            return this.deserialize(value, clasz);
        } finally {
            redisTemplate.unwatch();
        }
    }

    @Override
    public <T> T rightPop(KeyGenerator generator, Class<T> clasz) {
        return this.execPop(generator, false, clasz, null);
    }

    @Override
    public <T> T blockRightPop(KeyGenerator generator, Class<T> clasz, long timeout, TimeUnit unit) {
        return blockExecPop(generator, false, clasz, null, timeout, unit);
    }

    @Override
    public <T> T rightPop(KeyGenerator generator, TypeReference<T> typeReference) {
        return this.execPop(generator, false, null, typeReference);
    }

    @Override
    public <T> List<T> leftIndex(KeyGenerator generator, long index, Class<T> clasz) {
        return this.range(generator, 0, index, clasz);
    }

    @Override
    public <T> List<T> leftIndex(KeyGenerator generator, long index, TypeReference<T> typeReference) {
        return this.range(generator, 0, index, typeReference);
    }

    @Override
    public <T> T index(KeyGenerator generator, long index, Class<T> clasz) {
        Object value = operations.index(generator.gen(), index);
        return this.deserialize(value, clasz);
    }

    @Override
    public <T> T index(KeyGenerator generator, long index, TypeReference<T> typeReference) {
        Object value = operations.index(generator.gen(), index);
        return (T) getSerializer(typeReference).deserialize((byte[]) value);
    }

    @Override
    public <T> List<T> range(KeyGenerator generator, long start, long end, Class<T> clasz) {
        List<T> result = new ArrayList<>();

        List<Object> values = operations.range(generator.gen(), start, end);
        if (!CollectionUtils.isEmpty(values)) {
            RedisSerializer serializer = getSerializer(clasz);
            values.forEach(value -> result.add((T) serializer.deserialize((byte[]) value)));
        }
        return result;
    }

    @Override
    public <T> List<T> range(KeyGenerator generator, long start, long end, TypeReference<T> typeReference) {
        List<T> result = new ArrayList<>();

        List<Object> values = operations.range(generator.gen(), start, end);
        if (!CollectionUtils.isEmpty(values)) {
            RedisSerializer serializer = getSerializer(typeReference);
            values.forEach(value -> result.add((T) serializer.deserialize((byte[]) value)));
        }
        return result;
    }

    @Override
    public void leftTrim(KeyGenerator generator, long start, long end) {
        String key = generator.gen();
        try {
            redisTemplate.watch(key);
            redisTemplate.multi();

            operations.trim(generator.gen(), start, end);
        } finally {
            redisTemplate.unwatch();
        }
    }

}
