package com.bxm.newidea.component.redis.utils;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.redis.impl.DistributedLockImpl;
import com.google.common.math.LongMath;
import lombok.extern.slf4j.Slf4j;

import java.util.concurrent.TimeUnit;

/**
 * 令牌桶限流器
 *
 * @author: Meng.Liu
 * @date: 2018/11/12 下午4:31
 */
@Slf4j
public class RateLimiter {

    /**
     * redis key
     */
    private KeyGenerator key;

    /**
     * redis分布式锁的key
     *
     * @return
     */
    private String lockKey;

    /**
     * 每秒存入的令牌数
     */
    private Double permitsPerSecond;

    /**
     * 最大存储maxBurstSeconds秒生成的令牌
     */
    private Integer maxBurstSeconds;

    /**
     * 分布式同步锁
     */
    private DistributedLockImpl syncLock;

    private RedisStringAdapter redisStringAdapter;

    RateLimiter(KeyGenerator key,
                Double permitsPerSecond,
                Integer maxBurstSeconds,
                DistributedLockImpl syncLock,
                RedisStringAdapter redisStringAdapter) {
        this.key = key;
        this.lockKey = key.gen();
        this.permitsPerSecond = permitsPerSecond;
        this.maxBurstSeconds = maxBurstSeconds;
        this.syncLock = syncLock;
        this.redisStringAdapter = redisStringAdapter;
    }

    /**
     * 生成并存储默认令牌桶
     *
     * @return 令牌桶
     */
    private RedisPermits putDefaultPermits() {
        this.lock();
        try {
            RedisPermits obj = redisStringAdapter.get(key, RedisPermits.class);
            if (null == obj) {
                RedisPermits permits = new RedisPermits(permitsPerSecond, maxBurstSeconds);
                redisStringAdapter.set(key, permits, permits.expires());
                return permits;
            }
            return obj;
        } finally {
            this.unlock();
        }

    }

    /**
     * 令牌获取之前加锁
     */
    private void lock() {
        syncLock.lock(lockKey, 10, TimeUnit.SECONDS);
    }

    /**
     * 消费令牌后解锁
     */
    private void unlock() {
        syncLock.unlock(lockKey);
    }

    /**
     * 获取令牌桶，获取之前先获得锁
     */
    private RedisPermits getPermits() {
        RedisPermits obj = redisStringAdapter.get(key, RedisPermits.class);
        if (null == obj) {
            return putDefaultPermits();
        }
        return obj;
    }

    /**
     * 更新令牌桶
     *
     * @param permits 令牌桶
     */
    private void setPermits(RedisPermits permits) {
        redisStringAdapter.set(key, permits, permits.expires());
    }

    /**
     * 等待直到获取指定数量的令牌
     *
     * @param tokens 请求的令牌数量
     * @return 等待时长
     * @throws InterruptedException 线程异常中断
     */
    public Long acquire(Long tokens) throws InterruptedException {
        long milliToWait = this.reserve(tokens);
        log.info("acquire {} for {} ms", tokens, milliToWait);
        Thread.sleep(milliToWait);
        return milliToWait;
    }

    /**
     * 等待直到获取到一个令牌
     *
     * @return 等待时长
     * @throws InterruptedException 线程异常中断
     */
    public long acquire() throws InterruptedException {
        return acquire(1L);
    }

    /**
     * 等待指定的时长尝试获取指定数量的令牌
     *
     * @param tokens  要获取的令牌数
     * @param timeout 获取令牌等待的时间，负数被视为0
     * @return true表示获取成功，false表示等待获取超时
     */
    public Boolean tryAcquire(Long tokens, Long timeout, TimeUnit unit) throws InterruptedException {
        long timeoutMicros = Math.max(unit.toMillis(timeout), 0);
        checkTokens(tokens);
        Long milliToWait;
        try {
            this.lock();
            if (!this.canAcquire(tokens, timeoutMicros)) {
                return false;
            } else {
                milliToWait = this.reserveAndGetWaitLength(tokens);
            }
        } finally {
            this.unlock();
        }
        Thread.sleep(milliToWait);
        return true;
    }

    /**
     * 等待指定的时长尝试获取一个令牌
     *
     * @param timeout 获取令牌等待的时间，负数被视为0
     * @return true表示获取成功，false表示等待获取超时
     */
    public Boolean tryAcquire(Long timeout, TimeUnit unit) throws InterruptedException {
        return tryAcquire(1L, timeout, unit);
    }

    private long redisNow() {
        return System.currentTimeMillis();
    }

    /**
     * 获取令牌n个需要等待的时间
     *
     * @param tokens 获取的令牌数量
     * @return 需要等待的时间
     */
    private long reserve(Long tokens) {
        this.checkTokens(tokens);
        try {
            this.lock();
            return this.reserveAndGetWaitLength(tokens);
        } finally {
            this.unlock();
        }
    }

    /**
     * 校验token值
     *
     * @param tokens 获取数量是否正确
     */
    private void checkTokens(Long tokens) {
        if (tokens < 0) {
            throw new IllegalArgumentException("Requested tokens " + tokens + " must be positive");
        }
    }

    /**
     * 在等待的时间内是否可以获取到令牌
     *
     * @param tokens        令牌数量
     * @param timeoutMillis 超时时间
     * @return true表示可获取
     */
    private Boolean canAcquire(Long tokens, Long timeoutMillis) {
        return queryEarliestAvailable(tokens) - timeoutMillis <= 0;
    }

    /**
     * 返回获取{tokens}个令牌最早可用的时间
     *
     * @param tokens 令牌数量
     * @return 最早可用时间
     */
    private Long queryEarliestAvailable(Long tokens) {
        long n = redisNow();
        RedisPermits permit = this.getPermits();
        permit.reSync(n);
        // 可以消耗的令牌数
        long storedPermitsToSpend = Math.min(tokens, permit.getStoredPermits());
        // 需要等待的令牌数
        long freshPermits = tokens - storedPermitsToSpend;
        // 需要等待的时间
        long waitMillis = freshPermits * permit.getIntervalMillis();
        return LongMath.saturatedAdd(permit.getNextFreeTicketMillis() - n, waitMillis);
    }

    /**
     * 预定@{tokens}个令牌并返回所需要等待的时间
     *
     * @param tokens 获取令牌数量
     * @return 需要等待的时长
     */
    private Long reserveAndGetWaitLength(Long tokens) {
        long n = redisNow();
        RedisPermits permit = this.getPermits();
        permit.reSync(n);
        // 可以消耗的令牌数
        long storedPermitsToSpend = Math.min(tokens, permit.getStoredPermits());
        // 需要等待的令牌数
        long freshPermits = tokens - storedPermitsToSpend;
        // 需要等待的时间 = 等待制造的令牌 * 生产速度
        long waitMillis = freshPermits * permit.getIntervalMillis();

        //设置下次生产的时间
        permit.setNextFreeTicketMillis(LongMath.saturatedAdd(permit.getNextFreeTicketMillis(), waitMillis));
        //重新设置令牌桶中存储的令牌数量
        permit.setStoredPermits(permit.getStoredPermits() - storedPermitsToSpend);
        //更新本地存储令牌桶
        this.setPermits(permit);
        return permit.getNextFreeTicketMillis() - n;
    }
}