package com.bxm.newidea.component.schedule.rpc;

import com.alibaba.fastjson.JSON;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.config.ExecutorParam;
import com.bxm.newidea.component.schedule.config.XxlJobConfiguration;
import com.bxm.newidea.component.schedule.constant.CommonConstant;
import com.bxm.newidea.component.schedule.task.ScheduleTask;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.boot.context.event.ApplicationReadyEvent;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.ApplicationListener;

import java.util.Map;

@Slf4j
public class ScheduleServiceImpl implements ScheduleService, ApplicationContextAware, ApplicationListener<ApplicationReadyEvent> {

    private XxlJobConfiguration xxlJobConfiguration;

    private ScheduleFeignService scheduleFeignService;

    public ScheduleServiceImpl(XxlJobConfiguration xxlJobConfiguration, ScheduleFeignService scheduleFeignService) {
        this.xxlJobConfiguration = xxlJobConfiguration;
        this.scheduleFeignService = scheduleFeignService;
    }

    private ApplicationContext applicationContext;

    @Override
    public boolean push(ScheduleTask task) {
        if (validate(task)) {
            //临时性任务要做更新
            if (task.onceTask()) {
                scheduleFeignService.update(task.taskName(),
                        task.cron(),
                        task.description(),
                        xxlJobConfiguration.getAppName(),
                        CommonConstant.HANDLER_NAME,
                        buildParam(task),
                        task.routeStrategy().name(),
                        task.blockStrategy().name(),
                        task.author());
            } else {
                //永久性任务只做新增
                scheduleFeignService.add(task.taskName(),
                        task.cron(),
                        task.description(),
                        xxlJobConfiguration.getAppName(),
                        CommonConstant.HANDLER_NAME,
                        buildParam(task),
                        task.routeStrategy().name(),
                        task.blockStrategy().name(),
                        task.author());
            }
            return true;
        }
        return false;
    }

    @Override
    public boolean remove(String taskName) {
        if (StringUtils.isNotBlank(taskName)) {
            scheduleFeignService.remove(taskName);
            return true;
        }
        return false;
    }

    private String buildParam(ScheduleTask task) {
        return JSON.toJSONString(new ExecutorParam(task.callback().beanName(), task.callbackParam()));
    }

    private boolean validate(ScheduleTask task) {
        if (StringUtils.isBlank(task.taskName())) {
            log.error("任务名未配置");
            return false;
        }
        if (null == task.cron()) {
            log.error("任务未设置执行时间");
            return false;
        }
        if (null == task.callback()) {
            log.error("任务回调服务类未设置");
            return false;
        }
        if (!applicationContext.containsBean(task.callback().beanName())) {
            log.error("任务回调服务类未设置到Spring上下文");
            return false;
        }

        return true;
    }

    @Override
    public void onApplicationEvent(ApplicationReadyEvent event) {
        //获取spring中配置的调度任务
        Map<String, ScheduleTask> taskMap = applicationContext.getBeansOfType(ScheduleTask.class);
        if (taskMap.size() > 0) {
            taskMap.values().forEach(this::push);
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}

