package com.bxm.newidea.component.filter;

import com.bxm.newidea.component.annotations.LogicFilterBean;
import com.bxm.newidea.component.exception.ExcutorException;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.core.OrderComparator;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 过滤链执行，带有业务逻辑判断，在执行的过程中会中断
 * 适用场景：
 * 参见责任链模式
 *
 * @author liujia
 * @date 1/13/21 6:23 PM
 **/
@Component
@Slf4j
public class LogicFilterChainExecutor implements ApplicationRunner {

    private Map<String, List<ILogicFilter>> groupFilterMap = new HashMap<>();

    /**
     * 执行指定业务分组的过滤器
     * 过滤器的执行器
     *
     * @param group   业务分组
     * @param context 过滤器执行上下文
     */
    public <T> void doFilter(String group, T context) {
        List<ILogicFilter> filters = groupFilterMap.get(group);

        if (null == filters) {
            return;
        }

        try {
            LogicFilterChain<T> filterChain = new LogicFilterChain<>(filters);
            filterChain.filter(context);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            throw new ExcutorException("过滤器执行失败,策略分组：" + group, e);
        }
    }

    @Override
    public void run(ApplicationArguments args) {
        for (ILogicFilter filter : SpringContextHolder.getBeans(ILogicFilter.class)) {
            LogicFilterBean annotation = AnnotationUtils.findAnnotation(filter.getClass(), LogicFilterBean.class);
            if (null == annotation) {
                log.warn("[{}]未提供FilterBean注解", filter.getClass().getSimpleName());
                continue;
            }

            List<ILogicFilter> filters = groupFilterMap.get(annotation.group());
            if (filters == null) {
                filters = new ArrayList<>();
            }
            filters.add(filter);
            groupFilterMap.put(annotation.group(), filters);
        }

        for (List<ILogicFilter> rules : groupFilterMap.values()) {
            OrderComparator.sort(rules);
        }
    }
}
