package com.bxm.newidea.component.strategy;

import com.bxm.newidea.component.annotations.StrategyBean;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.core.OrderComparator;
import org.springframework.core.annotation.AnnotationUtils;
import org.springframework.stereotype.Component;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 带有返回值的策略执行器
 *
 * @author liujia
 * @date 1/13/21 6:03 PM
 **/
@Component
@Slf4j
public class ReturnedStrategyExecutor implements ApplicationRunner, ApplicationContextAware {

    private Map<String, List<IReturnedStrategy>> groupStrategyMap = new HashMap<>();

    private ApplicationContext applicationContext;

    /**
     * 执行业务逻辑分组内匹配的策略，支持多策略匹配与单策略匹配
     *
     * @param groupName 业务逻辑分组
     * @param param     策略执行上下文
     * @return 执行结果, 如果未匹配则可能返回null
     */
    @SuppressWarnings("unchecked")
    public <T, P> T execute(String groupName, P param) {
        List<IReturnedStrategy> strategies = groupStrategyMap.get(groupName);

        if (null == strategies) {
            log.error("不存在[{}]对应的策略逻辑分组", groupName);
            return null;
        }

        for (IReturnedStrategy strategy : strategies) {
            if (strategy.match(param)) {
                return (T) strategy.execute(param);
            }
        }

        log.info("[{}][{}]没有匹配到任何策略", groupName, param);

        return null;
    }

    /**
     * 执行业务逻辑分组内匹配的策略，支持多策略匹配与单策略匹配
     *
     * @param groupName    业务逻辑分组
     * @param strategyName 策略名称
     * @param param        策略执行上下文
     * @return 执行结果, 如果未匹配则可能返回null
     */
    @SuppressWarnings("unchecked")
    public <T, P> T execute(String groupName, String strategyName, P param) {
        List<IReturnedStrategy> strategies = groupStrategyMap.get(groupName);

        if (null == strategies) {
            log.error("不存在[{}]对应的策略逻辑分组", groupName);
            return null;
        }

        for (IReturnedStrategy strategy : strategies) {
            boolean match = (strategyName != null && strategyName.equalsIgnoreCase(strategy.name()))
                    || strategy.match(param);
            if (match) {
                return (T) strategy.execute(param);
            }
        }

        log.info("[{}]-[{}]-[{}]没有匹配到任何策略", groupName, strategyName, param);

        return null;
    }

    @Override
    public void run(ApplicationArguments args) {
        for (IReturnedStrategy strategy : applicationContext.getBeansOfType(IReturnedStrategy.class).values()) {
            StrategyBean annotation = AnnotationUtils.findAnnotation(strategy.getClass(), StrategyBean.class);
            if (null == annotation) {
                log.warn("[{}]未提供StrategyBean注解", strategy.getClass().getSimpleName());
                continue;
            }

            List<IReturnedStrategy> strategies = groupStrategyMap.get(annotation.group());
            if (strategies == null) {
                strategies = new ArrayList<>();
            }
            strategies.add(strategy);
            groupStrategyMap.put(annotation.group(), strategies);
        }

        for (List<IReturnedStrategy> strategies : groupStrategyMap.values()) {
            OrderComparator.sort(strategies);
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
