package com.bxm.component.tbk.order.api;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.httpclient.utils.OkHttpUtils;
import com.bxm.component.tbk.order.config.DuoMaiMeituanProperties;
import com.bxm.component.tbk.order.model.dto.MeituanOrderDetailInfo;
import com.bxm.component.tbk.order.model.dto.MeituanOrderInfo;
import com.bxm.component.tbk.order.model.param.QueryMeituanOrderListParam;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.List;
import java.util.Map;
import java.util.Objects;

@Service
@Slf4j
@AllArgsConstructor
public class MeituanOrderApi {

    private final DuoMaiMeituanProperties duoMaiMeituanProperties;

    private static final Map<String, String> HEARDS = Maps.newHashMap();


    public MeituanOrderDetailInfo getMeituanOrderDetailInfo(String orderSn) {

        if (StringUtils.isBlank(orderSn)) {
            log.warn("订单号为空，无法查询美团订单");
            return null;
        }

        try {

            Map<String, String> paramMap = Maps.newHashMap();
            // 接口密钥,
            paramMap.put("hash", duoMaiMeituanProperties.getSecret());
            // 请求动作，订单详情明细请填 action=query_detail
            paramMap.put("action", "query_detail");
            paramMap.put("order_sn", orderSn);
            paramMap.put("format", duoMaiMeituanProperties.getFormat());

            String result = OkHttpUtils.get(duoMaiMeituanProperties.getOrderDetailUrl(), paramMap, HEARDS);

            if (log.isDebugEnabled()) {
                log.debug("请求多麦联盟查询订单详情接口返回数据: {}", result);
            }

            List<MeituanOrderDetailInfo> meituanOrderDetailInfos;
            if (StringUtils.isBlank(result)
                    // 如果不是数组的json开头，说明查询失败了
                    || !result.startsWith("[")
                    || CollectionUtils.isEmpty(meituanOrderDetailInfos = JSON.parseArray(result,
                    MeituanOrderDetailInfo.class))) {
                log.warn("请求多麦联盟查询订单详情失败: {}", result);
                return null;
            }

            MeituanOrderDetailInfo detailInfo = meituanOrderDetailInfos.get(0);
            log.info("根据订单号: {} 查询详情: {}", orderSn, JSON.toJSONString(detailInfo));
            return detailInfo;

        } catch (Exception e) {
            log.error("请求多麦联盟查询订单详情失败", e);
        }

        return null;
    }

    /**
     * 查询多麦联盟的订单列表接口
     * @param param 查询参数
     * @return 订单列表
     */
    public List<MeituanOrderInfo> queryList(QueryMeituanOrderListParam param) {

        try {

            // 秘钥
            param.setHash(duoMaiMeituanProperties.getSecret());

            Map<String, String> paramMap = build(param);

            if (log.isDebugEnabled()) {
                log.debug("请求参数: {} 生成的请求参数: {}", JSON.toJSONString(param), paramMap);
            }

            String res = OkHttpUtils.get(duoMaiMeituanProperties.getOrderListQueryUrl(), paramMap, HEARDS);

            if (log.isDebugEnabled()) {
                log.debug("请求多麦联盟查询订单列表返回数据: {}", res);
            }

            if (StringUtils.isBlank(res) || !res.startsWith("[")) {
                return Lists.newArrayList();
            }

            return JSON.parseArray(res, MeituanOrderInfo.class);
        } catch (Exception e) {
            log.error("请求多麦联盟查询订单列表失败", e);
        }

        return Lists.newArrayList();
    }

    private Map<String, String> build(QueryMeituanOrderListParam param) {
        JSONObject jsonObject = JSONObject.parseObject(JSON.toJSONString(param));

        Map<String, String> paramMap = Maps.newHashMap();

        jsonObject.forEach((key, value) -> {
            if (StringUtils.isNotBlank(key)
                    && Objects.nonNull(value)) {
                paramMap.put(key, Objects.toString(value));
            }
        });

        return paramMap;
    }
}
