package com.bxm.component.tbk.factory;

import com.bxm.component.tbk.config.TbkConfigProperties;
import com.bxm.newidea.component.enums.AppNameEnum;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.taobao.api.DefaultTaobaoClient;
import com.taobao.api.TaobaoClient;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.util.List;
import java.util.Map;
import java.util.Objects;

/**
 * 淘宝client 的整合类
 * @author Gonzo
 * @date  2020-05-29 10:42
 * @since 1.0.0
 */
@Slf4j
public class TaobaoClientFactory {

    /**
     * 根据配置参数(默认 or 配置)生成的淘宝client 实例
     */
    private static final Map<String, TaobaoClient> TAOBAO_CLIENT_MAP = Maps.newHashMap();

    /**
     * 默认淘宝配置
     */
    private static final List<TbkSiteConfig> DEFAULT_CONFIG = Lists.newArrayList();

    /**
     * 淘宝开放平台的请求地址
     * 正式环境 http://gw.api.taobao.com/router/rest
     *  带https https://eco.taobao.com/router/rest
     * 沙箱环境 http://gw.api.tbsandbox.com/router/rest
     *  带https https://gw.api.tbsandbox.com/router/rest
     *
     *  默认配置
     */
    private static final String TB_OPEN_URL;

    static {

        TB_OPEN_URL = "https://eco.taobao.com/router/rest";
        // 趣淘金
        DEFAULT_CONFIG.add(new TbkSiteConfig(AppNameEnum.FOSSICKER.name(), "27666243",
                "fcb41e6d06d5cea3d5e080c5e682d215"));
        // 万事通
        DEFAULT_CONFIG.add(new TbkSiteConfig(AppNameEnum.LOCAL_NEWS.name(), "27650304",
                "a28530f72a39d0270762d98ba1f2273b"));
    }

    /**
     * 根据配置初始化
     * @param properties
     */
    public TaobaoClientFactory(TbkConfigProperties properties) {

        if (Objects.isNull(properties.getTbkAppKeyConfigMap()) || properties.getTbkAppKeyConfigMap().isEmpty()
            || Objects.isNull(properties.getTbkAppSecretConfigMap()) || properties.getTbkAppSecretConfigMap().isEmpty()) {

            // 如果配置信息不存在 则使用默认信息
            DEFAULT_CONFIG.forEach(p -> {
                String url = TB_OPEN_URL;
                String siteId = p.getSiteId();
                String appKey = p.getAppKey();
                String appSecret = p.getAppSecret();

                log.info("使用默认配置初始化TaoBaoClient siteId: {} url: {}, appKey: {} appSecret: {}", siteId,
                        url, appKey, appSecret);

                TAOBAO_CLIENT_MAP.put(siteId,
                        new DefaultTaobaoClient(url, appKey, appSecret));
            });
            return;
        }

        // 根据配置信息初始化
        properties.getTbkAppKeyConfigMap().forEach((key, value) -> {
            String url = properties.getTbOpenUrl();
            String siteId = key;
            String appKey = value;
            String appSecret = properties.getTbkAppSecretConfigMap().get(key);

            log.info("初始化TaoBaoClient siteId: {}, url: {}, appKey: {} appSecret: {}", siteId,  url, appKey, appSecret);

            TAOBAO_CLIENT_MAP.put(siteId,
                    new DefaultTaobaoClient(url, appKey, appSecret));
        });
    }

    /**
     * 根据业务主体，获取对应的tao bao client 实例
     * @param appName {@link AppNameEnum}
     * @return tao bao client
     */
    public TaobaoClient getClient(String appName) {
        if (log.isDebugEnabled()) {
            log.debug("根据appName: {}  获取实例", appName);
        }
        return TAOBAO_CLIENT_MAP.get(appName);
    }


    /**
     * 淘宝客配置
     * @author Gonzo
     * @date  2020-05-29 11:44
     * @since 1.0.0
     */
    @Data
    static class TbkSiteConfig {
        /**
         * 业务主体
         */
        private String siteId;

        /**
         * appKey
         */
        private String appKey;

        /**
         * appSecret
         */
        private String appSecret;

        public TbkSiteConfig(String siteId, String appKey, String appSecret) {
            this.siteId = siteId;
            this.appKey = appKey;
            this.appSecret = appSecret;
        }
    }
}
