package org.thunlp.thulac.io;

import org.thunlp.thulac.data.TaggedWord;

import java.io.IOException;
import java.util.List;

/**
 * An interface used to handle the output from the segmentation program. The whole
 * handling process is based on lines, though its extending the
 * {@link IProgramStateListener} allows it to listen the starting and termination
 * events of the program, therefore implementations should also concentrate on lines.
 */
public interface IOutputHandler extends IProgramStateListener {
	/**
	 * Handles the {@link List} of {@link TaggedWord} generated by the segmentation
	 * program. Since one input line might be split into multiple line segments,
	 * this method might be invoked several times between a pair of
	 * {@link #handleLineStart()} and {@link #handleLineEnd()}. Traditionally, the
	 * param {@code word} of all the invocations of this methods between a pair of
	 * {@link #handleLineEnd()} and {@link #handleLineEnd()} come from the same line of
	 * input, and the output handler should output to the same line as well, however
	 * this is not compulsory.
	 *
	 * @param words
	 * 		The {@link List} of {@link TaggedWord} generated processing one line segment.
	 * @param segOnly
	 * 		Whether to output without tags.
	 * @param separator
	 * 		The separator between output words and tags.
	 */
	void handleLineSegment(List<TaggedWord> words, boolean segOnly, char separator)
			throws IOException;

	/**
	 * Called when an input line is obtained from {@link IInputProvider} and the
	 * segmentation program is about to begin breaking the line into segments. This
	 * method is basically for initializations, e.g., creating new line, etc.<br>
	 * This method is invoked before {@link #handleLineSegment(List, boolean, char)}.
	 */
	void handleLineStart() throws IOException;

	/**
	 * Called when segmentation of an input line is finished and the segmentation
	 * program is about to begin processing the next line. This method is basically for
	 * finalisation, e.g., flushing input of this line, etc.<br>
	 * This method is invoked after {@link #handleLineSegment(List, boolean, char)}.
	 */
	void handleLineEnd() throws IOException;
}
