package com.bxm.newidea.component.uuid.strategy;

import com.bxm.newidea.component.annotations.StrategyBean;
import com.bxm.newidea.component.strategy.IReturnedStrategy;
import com.bxm.newidea.component.tools.NetworkInterfaceManager;
import com.bxm.newidea.component.uuid.config.ComponentUUIDConfigurationProperties;
import com.bxm.newidea.component.uuid.config.SequenceConfigHolder;
import com.bxm.newidea.component.uuid.constant.SequenceConstant;
import com.bxm.newidea.component.uuid.utils.SystemClock;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.CuratorFrameworkFactory;
import org.apache.curator.retry.RetryUntilElapsed;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.data.Stat;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;

import java.util.List;

import static com.bxm.newidea.component.uuid.constant.SequenceConstant.ACQUIRE_CONFIG_GROUP;
import static com.bxm.newidea.component.uuid.constant.SequenceConstant.ZOOKEEPER_STRATEGY_NAME;

/**
 * 通过zookeeper进行workerID协调
 *
 * @author liujia
 * @date 6/4/21 6:03 PM
 **/
@StrategyBean(group = ACQUIRE_CONFIG_GROUP)
@Slf4j
public class ZookeeperAcquireConfigStrategy implements IReturnedStrategy<SequenceConfigHolder, Void>, ApplicationRunner {

    private ComponentUUIDConfigurationProperties properties;

    public ZookeeperAcquireConfigStrategy(ComponentUUIDConfigurationProperties properties) {
        this.properties = properties;
    }

    private CuratorFramework curatorFramework;

    private String identification;

    @Override
    public SequenceConfigHolder execute(Void param) {
        SequenceConfigHolder holder;
        long workerId = 0;

        try {
            String foreverPath = getForeverPath();

            Stat stat = curatorFramework.checkExists().forPath(foreverPath);

            // 不存在根节点，认为是首次创建,
            if (stat == null) {
                createNode();
            } else {
                // 获取所有的子节点，解析后获取当前节点的顺序作为workerId
                workerId = getWorkerId();
            }

            holder = SequenceConfigHolder.builder()
                    .workerId(workerId)
                    .build();
        } catch (Exception e) {
            log.error("连接或访问zookeeper出错，采用备用方案。" + e.getMessage(), e);

            // 尝试读取本地文件
            holder = SequenceConfigHolder.builder()
                    .workerId(0L)
                    .build();
        }

        log.info("通过zookeeper获取workerId：[{}],path:[{}]", workerId, getNodePath() + workerId);

        return holder;
    }

    private long getWorkerId() throws Exception {
        List<String> keys = curatorFramework.getChildren().forPath(getForeverPath());

        String identification = getIdentification();

        for (String nodeKey : keys) {
            String[] split = StringUtils.split(nodeKey, "-");
            // 根据唯一标识查找是否存在对应的节点
            if (identification.equals(split[0])) {
                updateNodeData(nodeKey);
                return Long.valueOf(split[1]);
            }
        }

        // 不存在对应的节点，则创建新的节点
        String nodeKey = createNode();
        String[] split = StringUtils.split(nodeKey, "-");

        return Long.valueOf(split[1]);
    }

    private String createNode() throws Exception {
        String nodePath = getNodePath();

        log.info("创建存储uuid workerId的nodePath：{}", nodePath);

        return curatorFramework.create()
                .creatingParentsIfNeeded()
                .withMode(CreateMode.PERSISTENT_SEQUENTIAL)
                .forPath(nodePath, buildData());
    }

    private void updateNodeData(String nodeName) throws Exception {
        String path = getForeverPath() + "/" + nodeName;
        curatorFramework.setData().forPath(path, buildData());
    }

    private String getNodePath() {
        return getForeverPath() + "/" + getIdentification() + "-";
    }

    private String getForeverPath() {
        return SequenceConstant.ZK_PATH_PREFIX + "/" + properties.getLeafName();
    }

    private String getIdentification() {
        if (null == identification) {
            this.identification = NetworkInterfaceManager.INSTANCE.getLocalHostAddress() + ":" + properties.getLeafPort();
        }
        return this.identification;
    }

    private byte[] buildData() {
        return String.valueOf(SystemClock.now()).getBytes();
    }

    @Override
    public String name() {
        return ZOOKEEPER_STRATEGY_NAME;
    }

    @Override
    public void run(ApplicationArguments args) {
        if (ZOOKEEPER_STRATEGY_NAME.equalsIgnoreCase(properties.getAcquireConfigStrategy()) && null != properties.getZookeeperAddress()) {
            RetryUntilElapsed retry = new RetryUntilElapsed(1000, 4);
            curatorFramework = CuratorFrameworkFactory.builder()
                    .retryPolicy(retry)
                    .connectString(properties.getZookeeperAddress())
                    .connectionTimeoutMs(3000)
                    .sessionTimeoutMs(5000)
                    .build();

            curatorFramework.start();
        }
    }
}
