package com.bxm.newidea.component.mvc;

import com.bxm.newidea.component.exceptions.BusinessException;
import com.bxm.newidea.component.util.WebUtils;
import com.bxm.newidea.component.vo.ResponseJson;
import com.google.common.base.Joiner;
import lombok.extern.slf4j.Slf4j;
import org.springframework.context.support.DefaultMessageSourceResolvable;
import org.springframework.validation.BindingResult;
import org.springframework.validation.FieldError;
import org.springframework.web.HttpRequestMethodNotSupportedException;
import org.springframework.web.bind.MethodArgumentNotValidException;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.servlet.http.HttpServletRequest;
import javax.validation.ConstraintViolation;
import javax.validation.ConstraintViolationException;
import javax.validation.ValidationException;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 全局异常处理，统一返回http status为200
 * @author liujia 2018/3/30 21:20
 */
@ControllerAdvice
@ResponseBody
@Slf4j
public class CustomizeExceptionHandler {
    /**
     * 处理业务异常
     */
    @ExceptionHandler(BusinessException.class)
    public ResponseJson handleBusinessException(BusinessException e, HttpServletRequest request) {
        log(request, e);
        return ResponseJson.badReqeuset(e.getMessage()).build();
    }

    /**
     * 参数异常
     */
    @ExceptionHandler(IllegalArgumentException.class)
    public ResponseJson handleBusinessException(IllegalArgumentException e, HttpServletRequest request) {
        log(request, e);
        return ResponseJson.badReqeuset("请求参数不符合API要求").build();
    }

    /**
     * 校验异常信息处理
     * @param e       异常信息
     * @param request 请求信息
     * @return 异常处理后返回json格式数据到请求方
     */
    @ExceptionHandler({MethodArgumentNotValidException.class, ValidationException.class})
    public ResponseJson handleValidException(Exception e, HttpServletRequest request) {
        String errorMsg = null;

        if (e instanceof ConstraintViolationException) {
            ConstraintViolationException violationException = (ConstraintViolationException) e;

            if (violationException.getConstraintViolations() != null) {
                List<String> errorMsgList = violationException.getConstraintViolations()
                        .stream()
                        .map(ConstraintViolation::getMessage)
                        .collect(Collectors.toList());

                errorMsg = Joiner.on(";").skipNulls().join(errorMsgList);
            }
        } else if (e instanceof MethodArgumentNotValidException) {
            MethodArgumentNotValidException validException = (MethodArgumentNotValidException) e;
            BindingResult bindingResult = validException.getBindingResult();

            List<FieldError> fieldErrors = bindingResult.getFieldErrors();
            List<String> errorMsgList = fieldErrors.stream()
                    .map(DefaultMessageSourceResolvable::getDefaultMessage)
                    .collect(Collectors.toList());

            errorMsg = Joiner.on(";").skipNulls().join(errorMsgList);
        }

        log.warn("uri:[{}],param:[{}]", request.getRequestURI(), WebUtils.getReqeustParam(request));
        log.warn(e.getMessage(), e);

        if (null == errorMsg) {
            errorMsg = e.getMessage();
        }

        return ResponseJson.badReqeuset(errorMsg).build();
    }

    /**
     * 方法不支持
     */
    @ExceptionHandler(HttpRequestMethodNotSupportedException.class)
    public ResponseJson handleBusinessException(HttpRequestMethodNotSupportedException e, HttpServletRequest request) {
        log(request, e);
        return ResponseJson.badReqeuset("请选择与API一致的RequestMethod").build();
    }

    /**
     * 处理全局异常，当具体的异常类型未匹配时
     */
    @ExceptionHandler(Exception.class)
    public ResponseJson handleException(Exception e, HttpServletRequest request) {
        log(request, e);
        //避免将服务器内部的错误直接显示到用户端
        return ResponseJson.serverError("请求失败，请稍后再试").build();
    }

    private void log(HttpServletRequest request, Exception e) {
        log.error("uri:[{}],param:[{}]", request.getRequestURI(), WebUtils.getReqeustParam(request));
        log.error(e.getMessage(), e);
    }
}
