package com.bxm.newidea.component.log;

import com.bxm.newidea.component.util.WebUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.ThreadContext;
import org.springframework.stereotype.Component;

import javax.servlet.*;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

/**
 * 日志上下文过滤器，用于设置当前请求环境的上下文，用于后续的日志记录与现实
 *
 * @author liujia 2018/3/30 15:20
 */
@Component
@WebFilter(filterName = "log4j2ContextFilter", urlPatterns = "/**")
@Slf4j
public class LogContextFilter implements Filter {

    @Override
    public void init(FilterConfig filterConfig) {
        if (log.isDebugEnabled()) {
            log.debug("init log context filter");
        }
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        HttpServletResponse response = (HttpServletResponse) servletResponse;

        //修改request，使得可以读取多次requestBody
        if (!WebUtils.isMultipartRequest(request) && !(request instanceof ContentCachingRequestWrapper)) {
            request = new ContentCachingRequestWrapper(request);
        }

        if (log.isDebugEnabled()) {
            String url = request.getRequestURI();
            String requestParam = WebUtils.getRequestParam(request);
            log.debug("请求地址：[{}]，请求参数：[{}]",
                    url,
                    requestParam);
        }

        //设置当前请求的IP到Thread中，方便记录时保持IP信息
        ThreadContext.put(LogConstant.REQUEST_IP, WebUtils.getIpAddr(request));
        chain.doFilter(request, response);
    }

    @Override
    public void destroy() {
        ThreadContext.clearAll();
    }

    /**
     * 保存请求中的requestBody
     */
    private class ContentCachingRequestWrapper extends HttpServletRequestWrapper {

        private byte[] body;

        private BufferedReader reader;

        private ServletInputStream inputStream;

        private ContentCachingRequestWrapper(HttpServletRequest request) throws IOException {
            super(request);
            loadBody(request);
        }

        private void loadBody(HttpServletRequest request) throws IOException {
            body = IOUtils.toByteArray(request.getInputStream());
            inputStream = new RequestCachingInputStream(body);
        }

        @Override
        public ServletInputStream getInputStream() throws IOException {
            return new RequestCachingInputStream(body);
        }

        @Override
        public BufferedReader getReader() throws IOException {
            if (reader == null) {
                reader = new BufferedReader(new InputStreamReader(inputStream, getCharacterEncoding()));
            }
            return reader;
        }

        private class RequestCachingInputStream extends ServletInputStream {

            private final ByteArrayInputStream inputStream;

            private RequestCachingInputStream(byte[] bytes) {
                inputStream = new ByteArrayInputStream(bytes);
            }

            @Override
            public int read() {
                return inputStream.read();
            }

            @Override
            public int read(byte[] b) throws IOException {
                return inputStream.read(b);
            }

            @Override
            public boolean isFinished() {
                return inputStream.available() == 0;
            }

            @Override
            public boolean isReady() {
                return true;
            }

            @Override
            public void setReadListener(ReadListener readlistener) {
            }

        }

    }
}
