package com.bxm.newidea.component.config;

import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.SwaggerConfig;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.RestController;
import springfox.documentation.builders.ApiInfoBuilder;
import springfox.documentation.builders.PathSelectors;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.service.Contact;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spring.web.plugins.Docket;

import java.util.List;
import java.util.Set;

/**
 * 封装swagger对外暴露的API，网关通过调用API获取对应的信息，实现免配置
 *
 * @author liujia
 * @date 9/8/21 9:38 AM
 **/
@RestController
public class ExposeSwaggerApiConfiguration {

    private List<SwaggerConfig> apiModuleArray = Lists.newArrayList();

    private List<SwaggerConfig> manageApiModuleArray = Lists.newArrayList();

    private List<SwaggerConfig> thirdApiModuleArray = Lists.newArrayList();

    private Set<String> allModuleSet = Sets.newHashSet();

    @Value("${spring.application.name}")
    private String applicationName;

    @GetMapping("/expose/swagger/fetch")
    public List<SwaggerConfig> apiModule(String type) {
        switch (type) {
            case "manage":
                return manageApiModuleArray;
            case "3rd":
                return thirdApiModuleArray;
            default:
                return apiModuleArray;
        }
    }

    /**
     * 手动添加对网关暴露的API名称
     *
     * @param apiModule API名称，在swagger的下拉选择中
     */
    protected void addApiModule(SwaggerConfig apiModule) {
        checkRepeat(apiModule.getGroupName());
        apiModuleArray.add(apiModule);
    }

    /**
     * 手动添加对网关暴露的运营后台API名称
     *
     * @param manageApiModule API名称，在swagger的下拉选择中
     **/
    protected void addManageApiModule(SwaggerConfig manageApiModule) {
        checkRepeat(manageApiModule.getGroupName());
        manageApiModuleArray.add(manageApiModule);
    }

    /**
     * 手动添加对网关暴露的第三方API名称
     *
     * @param thirdApiModule API名称，在swagger的下拉选择中
     **/
    protected void addThirdApiModule(SwaggerConfig thirdApiModule) {
        checkRepeat(thirdApiModule.getGroupName());
        thirdApiModuleArray.add(thirdApiModule);
    }

    private void checkRepeat(String apiModule) {
        if (!allModuleSet.add(apiModule)) {
            throw new IllegalArgumentException("不支持定义相同的Module名称，请重命名");
        }
    }

    /**
     * 暴露一个面向C端用户的API
     *
     * @return swagger docket构建者
     */
    protected DocketBuilder exposeApi() {
        return new DocketBuilder().api();
    }

    /**
     * 暴露一个面向运营端用户的API
     *
     * @return swagger docket构建者
     */
    protected DocketBuilder exposeManageApi() {
        return new DocketBuilder().manageApi();
    }

    /**
     * 暴露一个面向第三方的API
     *
     * @return swagger docket构建者
     */
    protected DocketBuilder exposeThirdApi() {
        return new DocketBuilder().thirdApi();
    }

    public class DocketBuilder {

        private String moduleName;

        private boolean api;

        private boolean manageApi;

        private boolean thirdApi;

        private String version;

        private String description;

        private String scanPackage;

        private String author = "匿名";

        private String authorEmail = "";

        private int order = 0;

        private DocketBuilder() {
        }

        public DocketBuilder moduleName(String moduleName) {
            this.moduleName = moduleName;
            return this;
        }

        public DocketBuilder version(String version) {
            this.version = version;
            return this;
        }

        public DocketBuilder description(String description) {
            this.description = description;
            return this;
        }

        public DocketBuilder scanPackage(String scanPackage) {
            this.scanPackage = scanPackage;
            return this;
        }

        public DocketBuilder author(String author) {
            this.author = author;
            return this;
        }

        public DocketBuilder authorEmail(String authorEmail) {
            this.authorEmail = authorEmail;
            return this;
        }

        public DocketBuilder order(int order) {
            this.order = order;
            return this;
        }

        /**
         * 当前是否为面向C端用户的API
         */
        DocketBuilder api() {
            this.api = true;
            return this;
        }

        /**
         * 当前是否为面向第三方的API
         */
        DocketBuilder thirdApi() {
            this.thirdApi = true;
            return this;
        }

        /**
         * 当前是否为面向运营端用户的API
         */
        DocketBuilder manageApi() {
            this.manageApi = true;
            return this;
        }

        public Docket build() {
            if (StringUtils.isBlank(moduleName)) {
                throw new IllegalArgumentException("moduleName必须配置");
            }

            if (StringUtils.isBlank(scanPackage)) {
                throw new IllegalArgumentException("scanPackage必须配置");
            }

            SwaggerConfig swaggerConfig = new SwaggerConfig();
            swaggerConfig.setGroupName(moduleName);
            swaggerConfig.setOrder(order);
            swaggerConfig.setVersion(version);
            swaggerConfig.setServiceName(applicationName);

            if (api) {
                addApiModule(swaggerConfig);
            }

            if (manageApi) {
                addManageApiModule(swaggerConfig);
            }

            if (thirdApi) {
                addThirdApiModule(swaggerConfig);
            }


            ApiInfo apiInfo = new ApiInfoBuilder()
                    .title(moduleName)
                    .description(description)
                    .contact(new Contact(author, "", authorEmail))
                    .version(this.version)
                    .build();

            Docket docket = new Docket(DocumentationType.SWAGGER_2)
                    .apiInfo(apiInfo)
                    .groupName(moduleName)
                    .select()
                    .apis(RequestHandlerSelectors.basePackage(scanPackage))
                    .paths(PathSelectors.any())
                    .build();

            docket.useDefaultResponseMessages(false);
            return docket;
        }
    }
}
