package com.bxm.newidea.component.log;


import com.bxm.newidea.component.util.WebUtils;
import org.apache.commons.io.IOUtils;
import org.apache.logging.log4j.ThreadContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Component;

import javax.servlet.*;
import javax.servlet.annotation.WebFilter;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletRequestWrapper;
import javax.servlet.http.HttpServletResponse;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStreamReader;

/**
 * 日志上下文过滤器，用于设置当前请求环境的上下文，用于后续的日志记录与现实
 * @author liujia 2018/3/30 15:20
 */
@Component
@WebFilter(filterName = "zlogContextFilter", urlPatterns = "/**")
public class LogContextFilter implements Filter {

    private static final Logger logger = LoggerFactory.getLogger(LogContextFilter.class);

    @Override
    public void init(FilterConfig filterConfig) {
        logger.debug("init log context filter");
    }

    @Override
    public void doFilter(ServletRequest servletRequest, ServletResponse servletResponse, FilterChain chain) throws IOException, ServletException {
        HttpServletRequest request = (HttpServletRequest) servletRequest;
        HttpServletResponse response = (HttpServletResponse) servletResponse;

        //设置当前请求的IP
        ThreadContext.put(LogConstant.REQUEST_IP, WebUtils.getIpAddr(request));

        //修改request，使得可以读取多次requestBody
        if (!(request instanceof ContentCachingRequestWrapper)) {
            request = new ContentCachingRequestWrapper(request);
        }

        chain.doFilter(request, response);
    }

    @Override
    public void destroy() {
        ThreadContext.clearAll();
    }

    /**
     * 保存请求中的requestBody
     */
    private class ContentCachingRequestWrapper extends HttpServletRequestWrapper {

        private byte[] body;

        private BufferedReader reader;

        private ServletInputStream inputStream;

        private ContentCachingRequestWrapper(HttpServletRequest request) throws IOException {
            super(request);
            loadBody(request);
        }

        private void loadBody(HttpServletRequest request) throws IOException {
            body = IOUtils.toByteArray(request.getInputStream());
            inputStream = new RequestCachingInputStream(body);
        }

        private byte[] getBody() {
            return body;
        }

        @Override
        public ServletInputStream getInputStream() throws IOException {
            if (inputStream != null) {
                return inputStream;
            }
            return super.getInputStream();
        }

        @Override
        public BufferedReader getReader() throws IOException {
            if (reader == null) {
                reader = new BufferedReader(new InputStreamReader(inputStream, getCharacterEncoding()));
            }
            return reader;
        }

        private class RequestCachingInputStream extends ServletInputStream {

            private final ByteArrayInputStream inputStream;

            private RequestCachingInputStream(byte[] bytes) {
                inputStream = new ByteArrayInputStream(bytes);
            }

            @Override
            public int read() throws IOException {
                return inputStream.read();
            }

            @Override
            public boolean isFinished() {
                return inputStream.available() == 0;
            }

            @Override
            public boolean isReady() {
                return true;
            }

            @Override
            public void setReadListener(ReadListener readlistener) {
            }

        }

    }
}
