package com.bxm.newidea.component.util;

import com.bxm.newidea.component.constant.EncodedConstant;
import com.bxm.newidea.component.jwt.constant.JwtContant;
import com.bxm.newidea.component.jwt.tools.JwtTokenUtil;
import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.web.context.request.RequestAttributes;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.io.UnsupportedEncodingException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.net.URLDecoder;
import java.util.Enumeration;
import java.util.List;
import java.util.Map;

/**
 * web 请求相关的常用工具
 * @author liujia 2018/3/30 15:40
 */
public class WebUtils {

    private static Logger logger = LoggerFactory.getLogger(WebUtils.class);

    private WebUtils() {
    }

    /**
     * 根据请求获取用户的真实IP
     * @param request 请求
     * @return 解析的IP
     */
    public static String getIpAddr(HttpServletRequest request) {
        String ipAddress = request.getHeader("x-forwarded-for");
        if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
            ipAddress = request.getHeader("Proxy-Client-IP");
        }
        if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
            ipAddress = request.getHeader("WL-Proxy-Client-IP");
        }
        if (ipAddress == null || ipAddress.length() == 0 || "unknown".equalsIgnoreCase(ipAddress)) {
            ipAddress = request.getRemoteAddr();
            if ("127.0.0.1".equals(ipAddress) || "0:0:0:0:0:0:0:1".equals(ipAddress)) {
                ipAddress = getLocalRealIp();
            }
        }
        // 对于通过多个代理的情况，第一个IP为客户端真实IP,多个IP按照','分割
        if (ipAddress != null && ipAddress.length() > 15) {
            // = 15
            if (ipAddress.indexOf(",") > 0) {
                ipAddress = ipAddress.substring(0, ipAddress.indexOf(","));
            }
        }
        return ipAddress;
    }

    /**
     * 获取本机的真实IP
     * @return 本机的真实IP
     */
    public static String getLocalRealIp() {
        // 本地IP，如果没有配置外网IP则返回它
        String localip = null;
        // 外网IP
        String netip = null;

        Enumeration<NetworkInterface> netInterfaces = null;

        try {
            netInterfaces = NetworkInterface.getNetworkInterfaces();
            InetAddress ip = null;
            // 是否找到外网IP
            boolean finded = false;
            while (netInterfaces.hasMoreElements() && !finded) {
                NetworkInterface ni = netInterfaces.nextElement();
                Enumeration<InetAddress> address = ni.getInetAddresses();
                while (address.hasMoreElements()) {
                    ip = address.nextElement();
                    if (!ip.isSiteLocalAddress()
                            && !ip.isLoopbackAddress()
                            && !ip.getHostAddress().contains(":")) {
                        // 外网IP
                        netip = ip.getHostAddress();
                        finded = true;
                        break;
                    } else if (ip.isSiteLocalAddress()
                            && !ip.isLoopbackAddress()
                            && !ip.getHostAddress().contains(":")) {
                        // 内网IP
                        localip = ip.getHostAddress();
                    }
                }
            }

            if (netip != null && !"".equals(netip)) {
                return netip;
            } else {
                return localip;
            }
        } catch (SocketException e) {
            logger.warn("解析本机IP出现异常", e);
        }
        return "127.0.0.1";
    }

    /**
     * 判断一个请求是否是ajax
     * @param request 请求
     * @return true表示为ajax
     */
    public static boolean isAjax(HttpServletRequest request) {
        return "XMLHttpRequest".equals(request.getHeader("X-Requested-With"));
    }

    /**
     * 拼接request中的参数，获得拼接后的字符串
     * @param request      请求
     * @param joinChar     连接字符串
     * @param exludeFields 不参与拼接的字段
     * @return 拼接结果
     */
    public static String joinRequestParam(HttpServletRequest request, String joinChar, String... exludeFields) {
        Map<String, String> paramMap = Maps.newTreeMap();

        Enumeration<String> paramNames = request.getParameterNames();

        try {
            while (paramNames.hasMoreElements()) {
                String paramName = paramNames.nextElement();

                if (ArrayUtils.contains(exludeFields, paramName)) {
                    continue;
                }

                paramMap.put(paramName, URLDecoder.decode(request.getParameter(paramName), EncodedConstant.UTF8));
            }
        } catch (UnsupportedEncodingException e) {
            logger.error(e.getMessage(), e);
        }

        List<String> keyValues = Lists.newArrayList();

        paramMap.forEach((key, value) -> {
            keyValues.add(key + "=" + value);
        });

        return Joiner.on(joinChar).skipNulls().join(keyValues);
    }

    /**
     * 获取当前请求的用户ID
     * 优先从token中获取，如果不存在则尝试获取请求中的userId参数
     * @return 如果请求中不包含用户信息则返回空字符串
     */
    public static String getCurrentUserId() {
        HttpServletRequest request = getRequest();
        if (null != request) {
            String token = request.getHeader(JwtContant.TOKEN_HEADER);
            if (StringUtils.isNotBlank(token)) {
                return JwtTokenUtil.parseToken(token);
            }

            return request.getParameter(JwtContant.USER_ID_PARAM_NAME);
        }
        return StringUtils.EMPTY;
    }

    /**
     * 获取当前请求对象
     * @return 如果当前不包含请求则返回null
     */
    public static HttpServletRequest getRequest() {
        RequestAttributes requestAttributes = RequestContextHolder.currentRequestAttributes();

        if (requestAttributes instanceof ServletRequestAttributes) {
            ServletRequestAttributes servletRequestAttributes = (ServletRequestAttributes) requestAttributes;
            return servletRequestAttributes.getRequest();
        }

        return null;
    }

}
