package com.bxm.newidea.component.wechat.extend;

import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import lombok.extern.slf4j.Slf4j;
import me.chanjar.weixin.mp.api.WxMpInMemoryConfigStorage;
import me.chanjar.weixin.mp.enums.TicketType;

import static com.bxm.newidea.component.wechat.constant.RedisKeyConstant.WEIXIN_MP_ACCESS_KEY;

/**
 * 基于redis的accessToken存储，解决分布式环境下相同账号的accessToken多次获取，产生的不一致问题
 */
@Slf4j
public class WxMpInRedisConfigStorage extends WxMpInMemoryConfigStorage {

    private static final String ACCESS_TOKEN_KEY = "wechat_access_token_";

    private static final String JSAPI_TICKET_KEY = "wechat_jsapi_ticket_";

    private static final String CARDAPI_TICKET_KEY = "wechat_cardapi_ticket_";

    private RedisStringAdapter redisStringAdapter;

    private String accessTokenKey;

    private String jsapiTicketKey;

    private String cardapiTicketKey;

    private CustomApacheHttpClientBuilder customApacheHttpClientBuilder;

    private KeyGenerator accessKeyCacheKey;

    public WxMpInRedisConfigStorage(RedisStringAdapter redisStringAdapter, CustomApacheHttpClientBuilder customApacheHttpClientBuilder) {
        this.redisStringAdapter = redisStringAdapter;
        this.customApacheHttpClientBuilder = customApacheHttpClientBuilder;
        setApacheHttpClientBuilder(customApacheHttpClientBuilder);
    }

    @Override
    public void setAppId(String appId) {
        super.setAppId(appId);
        accessKeyCacheKey = WEIXIN_MP_ACCESS_KEY.copy().appendKey(appId);
        this.accessTokenKey = ACCESS_TOKEN_KEY.concat(appId);
        this.jsapiTicketKey = JSAPI_TICKET_KEY.concat(appId);
        this.cardapiTicketKey = CARDAPI_TICKET_KEY.concat(appId);
    }

    private KeyGenerator getWxMpKey(String key) {
        return accessKeyCacheKey.appendKey(key);
    }

    @Override
    public String getAccessToken() {
        return redisStringAdapter.getString(getWxMpKey(this.accessTokenKey));
    }

    @Override
    public void expireAccessToken() {
        log.info("{}主动执行accessToken过期", appId);
        //token错误时，用于主动执行过期
        redisStringAdapter.remove(getWxMpKey(this.accessTokenKey));
        super.expireAccessToken();
    }

    @Override
    public boolean isAccessTokenExpired() {
        return !redisStringAdapter.hasKey(getWxMpKey(this.accessTokenKey));
    }

    @Override
    public synchronized void updateAccessToken(String accessToken, int expiresInSeconds) {
        redisStringAdapter.set(getWxMpKey(this.accessTokenKey), accessToken, expiresInSeconds - 200L);
    }

    @Override
    public String getTicket(TicketType ticketType) {
        String jsapiTicket = redisStringAdapter.getString(getWxMpKey(this.jsapiTicketKey));

        log.info("[微信{}:的票据:{}]", appId, jsapiTicket);

        return jsapiTicket;
    }

    @Override
    public boolean isTicketExpired(TicketType ticketType) {
        KeyGenerator wxMpKey = getWxMpKey(this.jsapiTicketKey);

        log.info("[微信{}:js的key:{}是否过期]", appId, wxMpKey);

        return !redisStringAdapter.hasKey(wxMpKey);
    }

    @Override
    public synchronized void updateTicket(TicketType type, String ticket, int expiresInSeconds) {
        KeyGenerator wxMpKey = getWxMpKey(this.jsapiTicketKey);
        log.info("[微信{}:js的key:{},票据：{}, 过期时间: {}]", appId, wxMpKey, ticket, expiresInSeconds - 200L);

        redisStringAdapter.set(wxMpKey, ticket, expiresInSeconds - 200L);
    }
}
