package com.bxm.newidea.component.schedule.builder;

import com.bxm.newidea.component.schedule.constant.XxljobRouteStrategyEnum;
import com.bxm.newidea.component.schedule.task.TaskCallback;
import com.bxm.newidea.component.tools.CronUtil;
import com.xxl.job.core.enums.ExecutorBlockStrategyEnum;

import java.util.Date;

/**
 * 通过构建者创建仅运行一次的任务
 *
 * @author liujia
 */
public class OnceTaskBuilder extends InnerTaskAttribute {

    private OnceTaskBuilder() {
    }

    private Date fireTime;

    /**
     * 构建一个一次性调度任务
     *
     * @param taskName 任务名称，如果需要删除，则必须保证唯一
     * @param fireTime 触发时间，不得小于当前时间
     * @param callback 回调接口，必须是注册到spring的bean
     * @return 一次性任务
     */
    public static OnceTaskBuilder builder(String taskName, Date fireTime, TaskCallback callback) {
        OnceTaskBuilder task = new OnceTaskBuilder();
        task.fireTime = fireTime;
        task.taskName = taskName;
        task.callback = callback;
        return task;
    }

    /**
     * 构建一个一次性运行任务
     *
     * @param fireTime 触发时间，不可小于当前时间
     * @param callback 回调接口，必须注册到spring
     * @return 一次性任务
     */
    public static OnceTaskBuilder builder(Date fireTime, TaskCallback callback) {
        OnceTaskBuilder task = new OnceTaskBuilder();
        task.fireTime = fireTime;
        task.callback = callback;
        return task;
    }

    public InnerOnceTask build() {
        InnerOnceTask task = new InnerOnceTask();
        if (null != this.fireTime) {
            task.cron = CronUtil.formatDateToCron(this.fireTime);
        }
        task.taskName = this.taskName;
        task.callback = this.callback;
        task.description = this.description;
        task.param = this.param;
        task.routeStrategy = this.routeStrategy;
        task.blockStrategy = this.blockStrategy;
        task.author = this.author;
        return task;
    }

    /**
     * 调度任务名称，用于在调度平台区分任务信息
     */
    public OnceTaskBuilder taskName(String taskName) {
        this.taskName = taskName;
        return this;
    }

    /**
     * 任务触发时间
     */
    public OnceTaskBuilder fireTime(Date fireTime) {
        this.fireTime = fireTime;
        return this;
    }

    /**
     * 任务描述，说明调度任务的作用
     */
    public OnceTaskBuilder description(String description) {
        this.description = description;
        return this;
    }

    /**
     * 设置调度任务的创建人
     */
    public OnceTaskBuilder author(String author) {
        this.author = author;
        return this;
    }

    /**
     * 调度任务的回调接口，在到达任务执行周期时回调
     *
     * @return 回调服务
     */
    public OnceTaskBuilder callback(TaskCallback callback) {
        this.callback = callback;
        return this;
    }

    /**
     * 回调时携带的参数，需要在创建调度任务时传递
     */
    public OnceTaskBuilder callbackParam(Object param) {
        this.param = param;
        return this;
    }

    /**
     * 任务路由策略，在执行节点存在多个时，决定如何选择执行点
     */
    public OnceTaskBuilder routeStrategy(XxljobRouteStrategyEnum routeStrategy) {
        this.routeStrategy = routeStrategy;
        return this;
    }

    /**
     * 阻塞策略，在任务执行时前一个任务尚未完成，后续的任务如何调度
     */
    public OnceTaskBuilder blockStrategy(ExecutorBlockStrategyEnum blockStrategy) {
        this.blockStrategy = blockStrategy;
        return this;
    }
}
