package com.bxm.newidea.component.schedule.config;

import cn.hutool.core.util.NumberUtil;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.schedule.AbstractOnceXxlJob;
import com.bxm.newidea.component.schedule.job.BaseJobHandler;
import com.bxm.newidea.component.schedule.job.IXxlJobHandlerDefinition;
import com.bxm.newidea.component.schedule.model.*;
import com.google.common.base.Splitter;
import com.xxl.job.core.biz.model.ReturnT;
import com.xxl.job.core.util.XxlJobRemotingUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

/**
 * xxljob执行器的额外扩展逻辑，包括对任务的额外处理
 * 1. 根据当前应用名称初始化，对应xxljob中的group概念，一个服务对应一个group
 * 2. 完成应用初始化后，开始初始化具体的定时任务，修改了管理端的save job的逻辑
 *
 * @author liujia
 * @date 11/7/21 8:53 AM
 **/
@Slf4j
public class XxlJobExecutorExtend implements ApplicationContextAware {

    private XxlJobConfigurationProperties properties;

    private String accessToken;

    private String firstAdminAddress;

    private Integer groupId;

    private ApplicationContext applicationContext;

    private ComponentXxlJobExecutor executor;

    XxlJobExecutorExtend(XxlJobConfigurationProperties properties) {
        this.properties = properties;
        this.accessToken = properties.getAccessToken();

        firstAdminAddress = Splitter.on(",").split(properties.getAdminAddresses()).iterator().next();
        if (!firstAdminAddress.endsWith("/")) {
            firstAdminAddress += "/";
        }
    }

    void init(ComponentXxlJobExecutor executor) {
        this.executor = executor;
        // 注册分组
        registryGroup();

        // 注册执行器
        if (groupId != null) {
            registryTask();
        }
    }

    private void registryGroup() {
        XxlJobGroup group = new XxlJobGroup();
        group.setAppname(properties.getAppName());
        group.setTitle(properties.getAppName());
        group.setAddressType(0);

        ReturnT response = XxlJobRemotingUtil.postBody(firstAdminAddress + "api/group", accessToken, 3, group, String.class);

        if (response.getCode() == ReturnT.SUCCESS_CODE) {
            Object responseObj = response.getContent();
            if (responseObj != null) {
                groupId = NumberUtil.parseInt(responseObj.toString());
            } else {
                log.error("xxljob服务端注册服务时没有返回groupId,无法启动当前模块");
            }

        } else {
            log.error("当前应用注册失败，定时任务相关模块未启动成功，错误消息：{}", response.getMsg());
        }
    }

    /**
     * 将所有继承了抽象类的job注册到xxl-job服务端
     * 对新增的逻辑进行了重构，会先判断是否为已经存在的任务
     */
    private void registryTask() {
        Collection<BaseJobHandler> handlerCollection = scanJob();

        List<XxlJobInfo> jobInfoList = handlerCollection.stream().map(this::convert).collect(Collectors.toList());

        ReturnT response = XxlJobRemotingUtil.postBody(firstAdminAddress + "api/addJobs", accessToken, 3, jobInfoList, String.class);
        if (response.getCode() == ReturnT.SUCCESS_CODE) {
            // 任务注册到执行器，在被调用的时候会进行初始化
            handlerCollection.forEach(jobInfo -> ComponentXxlJobExecutor.registJobHandler(jobInfo.executorHandler(), jobInfo));
        } else {
            log.error("任务注册失败，错误消息：{}", response.getMsg());
        }
    }

    public <T extends BaseOnceJobParam> void submitOnceJob(Class<? extends AbstractOnceXxlJob> onceXxlJobClass, JobPersistentObject param) {
        ReturnT response = XxlJobRemotingUtil.postBody(firstAdminAddress + "api/onceJob", accessToken, 3, param, String.class);
        if (response.getCode() != ReturnT.SUCCESS_CODE) {
            log.error("{}提交失败，请求参数：{},错误原因：{}",
                    onceXxlJobClass,
                    JSON.toJSONString(param),
                    response.getMsg());
        }
    }

    public boolean cancelOnceJob(Class<? extends AbstractOnceXxlJob> onceXxlJobClass, CancelOnceJobParam param) {
        ReturnT response = XxlJobRemotingUtil.postBody(firstAdminAddress + "api/cancelOnceJob", accessToken, 3, param, String.class);
        if (response.getCode() != ReturnT.SUCCESS_CODE) {
            log.error("{}取消失败，请求参数：{},错误原因：{}",
                    onceXxlJobClass,
                    JSON.toJSONString(param),
                    response.getMsg());
            return false;
        }
        Object content = response.getContent();
        if (null == content) {
            return false;
        }

        return Boolean.TRUE.equals(Boolean.parseBoolean(content.toString()));
    }

    private XxlJobInfo convert(IXxlJobHandlerDefinition definition) {
        XxlJobInfo jobInfo = new XxlJobInfo();
        jobInfo.setJobGroup(groupId);
        jobInfo.setJobDesc(definition.jobDesc());
        jobInfo.setGlueType("BEAN");

        jobInfo.setAlarmEmail(properties.getAlarmEmail());
        jobInfo.setAuthor(definition.author());

        jobInfo.setScheduleType(definition.scheduleType().name());
        jobInfo.setScheduleConf(definition.scheduleConf());

        jobInfo.setMisfireStrategy(definition.misfireStrategy().name());

        jobInfo.setExecutorHandler(definition.executorHandler());
        jobInfo.setExecutorParam(definition.executorParam());
        jobInfo.setExecutorBlockStrategy(definition.executorBlockStrategy().name());
        jobInfo.setExecutorRouteStrategy(definition.executorRouteStrategy().name());
        jobInfo.setExecutorFailRetryCount(definition.executorFailRetryCount());
        jobInfo.setExecutorTimeout(definition.executorTimeout());

        return jobInfo;
    }

    private Collection<BaseJobHandler> scanJob() {
        Map<String, BaseJobHandler> jobHandlerMap = applicationContext.getBeansOfType(BaseJobHandler.class);

        log.info("扫描到{}个定时任务，进行初始化配置", jobHandlerMap.size());
        return jobHandlerMap.values();
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }
}
