package com.bxm.newidea.component.schedule;

import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.schedule.config.ComponentXxlJobExecutor;
import com.bxm.newidea.component.schedule.config.XxlJobConfigurationProperties;
import com.bxm.newidea.component.schedule.config.XxlJobExecutorExtend;
import com.bxm.newidea.component.schedule.model.BaseOnceJobParam;
import com.bxm.newidea.component.schedule.model.CancelOnceJobParam;
import com.bxm.newidea.component.schedule.model.JobPersistentObject;
import com.google.common.base.Preconditions;
import com.xxl.job.core.handler.IJobHandler;
import lombok.extern.slf4j.Slf4j;

/**
 * 一次性任务的执行器
 * 不支持更新，需要取消后重新提交
 *
 * @author liujia
 * @date 11/7/21 12:18 AM
 **/
@Slf4j
public class OnceJobExecutor {

    private final XxlJobConfigurationProperties properties;

    private final XxlJobExecutorExtend xxlJobExecutorExtend;

    public OnceJobExecutor(XxlJobConfigurationProperties properties, XxlJobExecutorExtend xxlJobExecutorExtend) {
        this.properties = properties;
        this.xxlJobExecutorExtend = xxlJobExecutorExtend;
    }

    /**
     * 提交一次性任务
     *
     * @param onceXxlJobClass 一次性任务对应的执行类，该类必须是一个spring bean
     * @param param           执行参数，必填
     */
    public <T extends BaseOnceJobParam> void submit(Class<? extends AbstractOnceXxlJob> onceXxlJobClass, T param) {
        Preconditions.checkNotNull(onceXxlJobClass);
        Preconditions.checkNotNull(param.getJobId());
        Preconditions.checkNotNull(param.getFireDate());

        if (properties.isMock()) {
            log.debug("模拟状态，不予处理，任务参数：{}", JSON.toJSONString(param));
            return;
        }

        IJobHandler iJobHandler = ComponentXxlJobExecutor.loadJobHandler(onceXxlJobClass.getSimpleName());
        if (iJobHandler == null) {
            log.error("{}不存在对应的执行器，请检查是否将它添加到spring context中", onceXxlJobClass);
            return;
        }

        JobPersistentObject jobObject = new JobPersistentObject();
        jobObject.setAppName(properties.getAppName());
        jobObject.setExecutorName(getExecutorName(onceXxlJobClass));
        jobObject.setJobId(rebuildJobId(jobObject.getExecutorName(), param.getJobId()));
        jobObject.setFireDate(param.getFireDate().getTime());
        jobObject.setParamJson(JSON.toJSONString(param));

        xxlJobExecutorExtend.submitOnceJob(onceXxlJobClass, jobObject);
    }

    /**
     * 取消任务执行
     *
     * @param onceXxlJobClass 执行器的类名
     * @param jobId           任务ID
     */
    public boolean cancel(Class<? extends AbstractOnceXxlJob> onceXxlJobClass, String jobId) {
        Preconditions.checkNotNull(onceXxlJobClass);

        if (properties.isMock()) {
            log.debug("模拟状态，不予处理，任务ID：{}", jobId);
            return true;
        }

        CancelOnceJobParam param = new CancelOnceJobParam();
        param.setAppName(properties.getAppName());
        param.setExecutorName(getExecutorName(onceXxlJobClass));
        param.setJobId(rebuildJobId(param.getExecutorName(), jobId));

        return xxlJobExecutorExtend.cancelOnceJob(onceXxlJobClass, param);
    }

    private String rebuildJobId(String executorName, String originJobId) {
        return properties.getAppName() + "_" + executorName + "_" + originJobId;
    }

    private String getExecutorName(Class<? extends AbstractOnceXxlJob> onceXxlJobClass) {
        return onceXxlJobClass.getSimpleName();
    }
}
