package com.bxm.newidea.component.zk.service;

import lombok.extern.slf4j.Slf4j;
import org.apache.curator.framework.CuratorFramework;
import org.apache.curator.framework.recipes.locks.InterProcessMutex;
import org.springframework.stereotype.Component;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;

/**
 * 基于zookeeper实现的独占式分布式锁，如果没有依赖redis的情况，可以考虑使用
 *
 * @author liujia
 * @date 6/7/21 9:57 PM
 **/
@Component
@Slf4j
public class ZookeeperDistributedLock {

    private CuratorFramework curatorFramework;

    private ThreadLocal<Map<String, InterProcessMutex>> lockThreadLocal = new ThreadLocal<>();

    public ZookeeperDistributedLock(CuratorFramework curatorFramework) {
        this.curatorFramework = curatorFramework;
    }

    /**
     * 加锁
     *
     * @param resource 被锁住的资源描述，保证唯一
     * @return true表示获取锁成功
     */
    public boolean lock(String resource) {
        return lock(resource, 100, TimeUnit.MILLISECONDS);
    }

    /**
     * 加锁
     *
     * @param resource 被锁住的资源描述，保证唯一
     * @param time     获取锁的等待时间
     * @param timeUnit 等待时间的单位
     * @return true表示获取锁成功
     */
    public boolean lock(String resource, long time, TimeUnit timeUnit) {
        boolean result = false;

        try {
            InterProcessMutex lock = getLock(resource);

            result = lock.acquire(time, timeUnit);
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }

        if (log.isDebugEnabled()) {
            log.debug("给[{}]加锁，加锁结果：[{}]，锁定时间：[{}][{}]", resource, result, timeUnit.name(), time);
        }

        return result;
    }

    /**
     * 解锁，每次加锁必须对应一次解锁
     *
     * @param resource
     */
    public void unlock(String resource) {
        if (log.isDebugEnabled()) {
            log.debug("[{}]解锁", resource);
        }

        Map<String, InterProcessMutex> map = lockThreadLocal.get();

        if (null != map) {
            InterProcessMutex lock = map.get(resource);
            try {
                if (null != lock) {
                    if (lock.isOwnedByCurrentThread()) {
                        lock.release();
                        map.remove(resource);
                    } else {
                        log.info("解锁失败：{}", lock);
                    }
                }

                if (map.size() == 0) {
                    if (log.isDebugEnabled()) {
                        log.debug("清空threadLocal");
                    }
                    lockThreadLocal.remove();
                }
            } catch (Exception e) {
                log.error(e.getMessage(), e);
            }
        }
    }

    private InterProcessMutex getLock(String resource) {
        Map<String, InterProcessMutex> map = lockThreadLocal.get();
        if (map == null) {
            map = new ConcurrentHashMap<>(5);
            lockThreadLocal.set(map);
        }
        String lockNodeParent = "/component/lock/";
        String lockNode = lockNodeParent + resource;

        InterProcessMutex lock = map.get(resource);

        if (null == lock) {
            lock = new InterProcessMutex(curatorFramework, lockNode);
            map.put(resource, lock);
        }

        log.info("map size:{}", map.size());
        return lock;
    }
}
