package com.bxm.newidea.component.zk.service;

import com.bxm.newidea.component.zk.listener.NodeChangeListener;
import com.bxm.newidea.component.zk.listener.NodeChangeWithChildrenListener;
import org.apache.curator.framework.CuratorFramework;
import org.apache.zookeeper.CreateMode;
import org.apache.zookeeper.data.Stat;

import java.util.List;

/**
 * 对zookeeper提供适配接口实现，减少学习成本和简化调用
 *
 * @author liujia
 * @date 6/7/21 9:22 AM
 **/
public interface ZookeeperAdapter {

    /**
     * 变更当前默认的连接类，存在多个zookeeper连接或者有自定义zookeeper连接时使用
     * 将连接类（curatorFramework）设置到threadLocal中，仅限当前线程使用
     * 调用函数完毕后，需要最终调用{@link #releaseCuratorFramework()}
     *
     * @param curatorFramework 连接框架
     */
    void changeCuratorFramework(CuratorFramework curatorFramework);

    /**
     * 如果调用了{@link #changeCuratorFramework(CuratorFramework)}
     * 则需要释放设置的CuratorFramework
     * 防止TreadLocal未清理导致内存泄露，需要手动进行调用
     */
    void releaseCuratorFramework();

    /**
     * 创建持久化节点
     *
     * @param fullPath 节点路径，如果是多层级节点，会自动创建父节点
     * @return 创建的节点名称，不包含路径前缀，仅包含节点名称
     */
    String createNode(String fullPath);

    /**
     * 创建持久化节点，并写入节点数据
     *
     * @param fullPath 节点路径，如果是多层级节点，会自动创建父节点
     * @param data     节点数据
     * @return 创建的节点名称，不包含路径前缀，仅包含节点名称
     */
    <T> String createNode(String fullPath, T data);

    /**
     * 创建节点，并指定节点创建的模式，同时写入节点数据
     *
     * @param fullPath   节点路径，如果是多层级节点，会自动创建父节点
     * @param createMode 节点创建的模式，有临时节点、永久节点、永久有序节点等
     * @return 创建的节点名称，不包含路径前缀，仅包含节点名称
     */
    String createNode(String fullPath, CreateMode createMode);

    /**
     * 创建节点，并指定节点创建的模式，同时写入节点数据
     *
     * @param fullPath   节点路径，如果是多层级节点，会自动创建父节点
     * @param createMode 节点创建的模式，有临时节点、永久节点、永久有序节点等
     * @param data       节点写入的数据
     * @return 创建的节点名称，不包含路径前缀，仅包含节点名称
     */
    <T> String createNode(String fullPath, CreateMode createMode, T data);

    /**
     * 根据path删除指定的节点
     *
     * @param fullPath 删除节点的完整路径
     */
    void removeNode(String fullPath);

    /**
     * 根据path删除指定的节点，包含其子节点
     * 慎重调用，请确定是否有必要
     *
     * @param fullPath 删除节点的完整路径
     */
    void removeNodeWithChildren(String fullPath);

    /**
     * 设置指定节点的值
     *
     * @param fullPath 节点的完整地址
     * @param data     节点值
     * @return 节点的状态信息，如果节点不存在，则会返回null
     */
    <T> Stat setData(String fullPath, T data);

    /**
     * 设置指定节点的值
     *
     * @param fullPath 节点的完整地址
     * @param data     节点值
     * @param version  节点版本，用于乐观锁
     * @return 节点的状态信息，如果节点不存在或者版本号错误，则会返回null
     */
    <T> Stat setData(String fullPath, T data, int version);

    /**
     * 获取节点的所有子节点
     *
     * @param fullPath     节点完整地址
     * @param fillFullPath 是否返回子节点的完整地址，默认为false
     * @return 子节点名称（是否为完整地址，根据参数fullPath决定）,如果不存在则返回空数组
     */
    List<String> obtainChildren(String fullPath, boolean fillFullPath);

    /**
     * 获取节点存储的数据
     *
     * @param fullPath  节点完整路径
     * @param dataClass 节点存储数据的类型
     * @return 节点存储数据，如果无法解析，则返回null
     */
    <T> T obtainData(String fullPath, Class<T> dataClass);

    /**
     * 获取节点的状态信息
     *
     * @param fullPath 节点完整路径
     * @return 节点状态信息
     */
    Stat obtainStat(String fullPath);

    /**
     * 检查节点是否存在
     *
     * @param fullPath 节点完整地址
     * @return true表示节点已经存在
     */
    boolean checkExists(String fullPath);

    /**
     * 添加节点的监听器
     *
     * @param fullPath 监听节点的完整地址
     * @param listener 监听器,需要通过泛型明确的指定存储的值类型
     */
    void addListener(String fullPath, NodeChangeListener listener);

    /**
     * 添加节点监听器，可监听子节点
     *
     * @param fullPath 节点的完整地址
     * @param listener 监听器
     * @param self     true表示包含监听自身，false表示仅监听子节点。
     *                 如果为false，监听器获取不到对应节点的值，只能监听事件类型
     */
    void addListener(String fullPath, NodeChangeWithChildrenListener listener, boolean self);
}
