package com.bxm.localnews.integration;

import com.bxm.egg.common.url.ProtocolFactory;
import com.bxm.egg.message.facade.service.MessageFacadeService;
import com.bxm.egg.mq.common.constant.TemplateTypeEnum;
import com.bxm.egg.mq.common.model.dto.PushMessage;
import com.bxm.egg.mq.common.model.dto.PushPayloadInfo;
import com.bxm.egg.mq.common.model.dto.PushReceiveScope;
import com.bxm.foundation.base.facade.service.DomainFacadeService;
import com.bxm.localnews.news.model.bo.ForumPostDetailBO;
import com.bxm.localnews.news.model.entity.ForumPostEntity;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import static com.bxm.egg.mq.common.constant.PushMessageEnum.JUMP_TYPE;

@Service
@Slf4j
public class PushMsgService {

    private DomainFacadeService domainIntegrationService;

    private MessageFacadeService messageFacadeService;

    public PushMsgService(InnerServiceIntegrationService innerServiceIntegrationService) {
        this.messageFacadeService = innerServiceIntegrationService.getMessageFacadeService();
        domainIntegrationService = innerServiceIntegrationService.getDomainFacadeService();
    }

    public void pushMsg(PushMessage message) {
        this.messageFacadeService.sendPushMessage(message);
    }

    private String getPostProtocolUrl(Long postId) {
        return ProtocolFactory.forumPost().app().postId(postId).build();
    }

    /**
     * 推送帖子被删除消息
     *
     * @param postEntity   帖子
     * @param targetPostId 推送要跳转的帖子id
     */
    public void pushPostDeleteMsg(ForumPostEntity postEntity, Long targetPostId) {
        // 抱歉，你的信息「....」已被管理员删除。为给圈友提供更优质的内容和社区氛围，我们会将一些内容违规或质量较低的信息作删除处理，感谢你的理解
        String title = postEntity.getBrief();
        title = title.length() > 8 ? title.substring(0, 8) : title;
        String content = "抱歉，你的信息「" + title + "」已被管理员删除。为给圈友提供更优质的内容和社区氛围，" +
                "我们会将一些内容违规或质量较低的信息作删除处理，感谢你的理解";
        PushPayloadInfo info = PushPayloadInfo.build(JUMP_TYPE);
        Long msgId = SequenceHolder.nextLongId();
        info.setMsgId(msgId);
        // 跳转帖子详情页，帖子ID：4000000000182903（正式环境）
        info.setProtocol(getPostProtocolUrl(targetPostId));
        info.addExtend("msgId", msgId);
        info.addExtend("postId", postEntity.getId());
        PushMessage message = PushMessage.build();
        message.setTitle("信息已被删除");
        message.setContent(content);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(postEntity.getUserId()));
        message.setPayloadInfo(info);

        pushMsg(message);

    }

    /**
     * 推送帖子被拒绝消息
     *
     * @param post         帖子
     * @param targetPostId 推送要跳转的帖子id
     */
    public void pushPostRejectMsg(ForumPostDetailBO post, Long targetPostId) {
        // 抱歉，你的信息「....」被处理为违规信息。为给圈友提供更优质的内容和社区氛围，我们会将一些内容违规或质量较低的信息作处理，感谢你的理解
        String title = post.getPostInfo().getBrief();
        title = title.length() > 8 ? title.substring(0, 8) : title;
        String content = "抱歉，你的信息「" + title + "」被处理为违规信息。为给圈友提供更优质的内容和社区氛围，" +
                "我们会将一些内容违规或质量较低的信息作处理，感谢你的理解";
        PushPayloadInfo info = PushPayloadInfo.build(JUMP_TYPE);
        Long msgId = SequenceHolder.nextLongId();
        info.setMsgId(msgId);
        // 跳转帖子详情页，帖子ID：4000000000182903（正式环境）
        info.setProtocol(getPostProtocolUrl(targetPostId));
        info.addExtend("msgId", msgId);
        info.addExtend("postId", post.getPostInfo().getId());
        PushMessage message = PushMessage.build();
        message.setTitle("你的信息已违规");
        message.setContent(content);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(post.getPostInfo().getUserId()));
        message.setPayloadInfo(info);

        pushMsg(message);
    }

    /**
     * 推送评论被删除和拒绝的消息
     *
     * @param reply        评论内容
     * @param userId       用户uid
     * @param targetPostId 跳转的帖子id
     */
    public void pushReplyDeleteAndRejectMsg(String reply, Long userId, Long targetPostId) {
        // 经举报和审核，你的评论「....」违反 本地蛋蛋佳管理规定，已被删除。感谢你的理解。
        String title = reply.length() > 8 ? reply.substring(0, 8) : reply;
        // 如果是图片评论
        title = StringUtils.isBlank(title) ? "查看图片" : title;

        String content = "经举报和审核，你的评论「" + title + "」违反蛋蛋佳管理规定，已被删除。感谢你的理解。";
        PushPayloadInfo info = PushPayloadInfo.build(JUMP_TYPE);
        Long msgId = SequenceHolder.nextLongId();
        info.setMsgId(msgId);
        // 跳转帖子详情页，帖子ID：4000000000182903（正式环境）
        info.setProtocol(getPostProtocolUrl(targetPostId));
        info.addExtend("msgId", msgId);
        info.addExtend("replyId", targetPostId);
        PushMessage message = PushMessage.build();
        message.setTitle("评论已被删除");
        message.setContent(content);
        message.setType(TemplateTypeEnum.NOTIFCTION);
        message.setPushReceiveScope(PushReceiveScope.pushSignle(userId));
        message.setPayloadInfo(info);

        pushMsg(message);
    }

}
