package com.bxm.localnews.news.comment;

import com.bxm.egg.common.vo.Json;
import com.bxm.localnews.news.model.dto.NewsReplyDTO;
import com.bxm.localnews.news.model.dto.NewsReplyDetailDTO;
import com.bxm.localnews.news.model.dto.NewsReplyListDTO;
import com.bxm.localnews.news.model.dto.ReplyDTO;
import com.bxm.localnews.news.model.param.NewsReplyAddParam;
import com.bxm.localnews.news.model.param.NewsReplyLikeParam;
import com.bxm.localnews.news.model.param.NewsReplyParam;
import com.bxm.localnews.news.model.param.UserReplyParam;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.model.vo.MyReplysVO;
import com.bxm.localnews.news.model.vo.NewsReply;
import com.bxm.localnews.news.vo.PageWarper;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.param.BasicParam;

import java.util.Date;

/**
 * 新闻回复接口 包括普通新闻回复和小视频回复
 */
public interface NewNewsReplyService {

    /**
     * 用途：获取新闻|小视频|帖子回复列表
     *
     * @param newsReplyParam 新闻|小视频|帖子传参
     * @return 返回评论实体，根据评论的分级分别显示不同的实体
     */
    PageWarper<NewsReplyDTO> queryReplyList(NewsReplyParam newsReplyParam, BasicParam basicParam);

    /**
     * 用途：用户发表评论
     *
     * @param newsReply  评论参数
     * @param basicParam 基础参数
     * @return
     */
    Json<ReplyDTO> doReply(NewsReplyAddParam newsReply, BasicParam basicParam);

    /**
     * 当评论类型是帖子时,保存帖子的评论记录到缓存
     *
     * @param userId   用户id
     * @param postId   帖子id
     * @param sendTime 评论时间
     */
    void saveForumPostReplyRecord(Long userId, Long postId, Date sendTime);

    /**
     * 用途：根据用户id获取用户的评论列表
     *
     * @param newsReplyParam
     * @return
     */
    PageWarper<MyReplysVO> selectMyReplys(UserReplyParam newsReplyParam, BasicParam basicParam);

    /**
     * 用途：删除评论
     *
     * @param replyId
     * @return
     */
    NewsReplyDTO delMyReply(Long replyId, Long userId, Long newsId, Byte isUserDelete);

    /**
     * 更新评论中用户头像以及昵称
     *
     * @param userId
     * @param nickname
     * @param headImg
     */
    void doTriggerUpdateInfo(Long userId, String nickname, String headImg);

    /**
     * 用途：用户对新闻的评论进行点赞后，都在服务端处理并输送至redis阻塞队列，等待
     * 消费端，每5秒钟消费一次,更新评论信息
     *
     * @param newsReplyLikeParam
     */
    Message doProduceNewsReplyLike(NewsReplyLikeParam newsReplyLikeParam);

    /**
     * 用途：对redis队列中的回复点赞进行消费
     */
    void doNewsReplyLikeConsume();

    /**
     * 用途：获取新闻|小视频|帖子评论详情
     *
     * @param replyId
     * @param userId
     * @param newsId
     * @return
     */
    NewsReplyDetailDTO getNewsReplyDetailDTO(Long replyId, Long userId, Long newsId, String areaCode, BasicParam param);

    /**
     * 用途：用户在评论后所做的一系列异步操作
     * 1.保存至用户评论中
     * 2.新闻不是根级评论时将根级评论互动值+3
     * 3.增加评论数
     * 4.推送消息，自己给自己评论不推送信息
     * 5.记录事件埋点
     * 6.如果评论内容当中涉及到@用户 则需要进行@用户的处理
     *
     * @param newsReply
     * @param newsReplyAddParam
     */
    void processWhenReply(NewsReply newsReply, NewsReplyAddParam newsReplyAddParam, Integer platform, String areaCode);

    /**
     * 当审核评论可被展示时 后续的处理
     *
     * @param newsReply 评论信息
     */
    void processWhenReplySuccess(NewsReply newsReply);

    /**
     * 给被@评论的人推送消息
     *
     * @param newsReply   评论内容
     * @param forumPostVo 帖子信息
     */
    void pushPostReplyATMessage(NewsReply newsReply, ForumPostVo forumPostVo);

    /**
     * 维护静态数据
     *
     * @param newsReply
     */
    void updateComment(NewsReply newsReply);

    /**
     * 历史数据评论处理
     *
     * @return
     */
    Boolean doDealHistoryReply();

    /**
     * 统计一段时间内贴子的评论数量,并发送推送
     */
    void statisticalPostReply();


    /**
     * 3.9.0需求
     * <p>
     * 评论列表 区分热门评论 + 普通评论
     * <p>
     * 热门评论逻辑：
     * 1- 运营设置的热门评论
     * 2- 点赞超过两个的评论
     *
     * @param newsReplyParam
     * @return 评论集合
     */
    NewsReplyListDTO replyList(NewsReplyParam newsReplyParam, BasicParam basicParam);


    /**
     * 用途：删除新闻下的评论
     *
     * @param id
     * @param newsId
     */
    void delReply(Long id, Long newsId);
}
