package com.bxm.localnews.news.comment.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.egg.common.utils.ResultUtil;
import com.bxm.egg.common.vo.Json;
import com.bxm.egg.mq.common.constant.PushMessageEnum;
import com.bxm.egg.user.enums.WarmRuleEnum;
import com.bxm.egg.user.param.UserWarmActionParam;
import com.bxm.foundation.base.facade.service.SensitiveWordFacadeService;
import com.bxm.localnews.integration.*;
import com.bxm.localnews.news.comment.NewNewsReplyService;
import com.bxm.localnews.news.comment.service.CommentFacadeService;
import com.bxm.localnews.news.config.ForumProperties;
import com.bxm.localnews.news.constant.MemoryCacheKey;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.convert.NewsReplyConver;
import com.bxm.localnews.news.detail.helper.ForumPostImageHelper;
import com.bxm.localnews.news.domain.AdminNewsReplyMapper;
import com.bxm.localnews.news.domain.NewsReplyLikeMapper;
import com.bxm.localnews.news.domain.NewsReplyMapper;
import com.bxm.localnews.news.domain.UserReplyMapper;
import com.bxm.localnews.news.dto.LocationDTO;
import com.bxm.localnews.news.enums.NewsConstant;
import com.bxm.localnews.news.enums.ReplyStatusEnum;
import com.bxm.localnews.news.enums.ReplyTypeEnum;
import com.bxm.localnews.news.event.CommentActionEvent;
import com.bxm.localnews.news.event.ReplyAccessApplicationEvent;
import com.bxm.localnews.news.event.UserActionEvent;
import com.bxm.localnews.news.facade.service.ForumPostInnerService;
import com.bxm.localnews.news.facade.service.ForumPostTotalFacadeService;
import com.bxm.localnews.news.factory.ExtendFactory;
import com.bxm.localnews.news.model.dto.*;
import com.bxm.localnews.news.model.entity.ForumPostTotalEntity;
import com.bxm.localnews.news.model.param.NewsReplyAddParam;
import com.bxm.localnews.news.model.param.NewsReplyLikeParam;
import com.bxm.localnews.news.model.param.NewsReplyParam;
import com.bxm.localnews.news.model.param.UserReplyParam;
import com.bxm.localnews.news.model.vo.*;
import com.bxm.localnews.news.thread.PostReplyThread;
import com.bxm.localnews.news.util.AtUserIdExtractUtils;
import com.bxm.localnews.news.util.FormPostContentUtil;
import com.bxm.localnews.news.vo.ForumReplyInfo;
import com.bxm.localnews.news.vo.PageWarper;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.enums.PlatformEnum;
import com.bxm.newidea.component.param.BasicParam;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.sync.core.CacheHolder;
import com.bxm.newidea.component.tools.DateBeautifyUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.eventbus.EventBus;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.jsoup.Jsoup;
import org.jsoup.nodes.Document;
import org.springframework.beans.BeanUtils;
import org.springframework.context.event.EventListener;
import org.springframework.core.task.AsyncTaskExecutor;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSONString;
import static com.bxm.localnews.news.util.ReplyUtils.processReplyContent;
import static com.bxm.localnews.news.utils.CommentUtil.filter;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * Created by Administrator on 2018/2/23 0023.
 */
@Slf4j
@Service
@Transactional(rollbackFor = Exception.class)
public class NewNewsReplyServiceImpl extends BaseService implements NewNewsReplyService {

    @Resource
    private NewsReplyMapper newsReplyMapper;

    @Resource
    private NewsReplyLikeMapper newsReplyLikeMapper;

    @Resource
    private RedisListAdapter redisListAdapter;

    @Resource
    private UserReplyMapper userReplyMapper;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private MessageService messageService;

    @Resource
    private NewsReplyConver newsReplyConver;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Resource
    private SensitiveWordFacadeService sensitiveWordFacadeService;

    @Resource
    private EventBus userActionEventBus;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @Resource
    private AsyncTaskExecutor asyncTaskExecutor;

    @Resource
    private CommentFacadeService commentFacadeService;

    @Resource
    private AdminNewsReplyMapper adminNewsReplyMapper;

    @Resource
    private ForumPostImageHelper forumPostImageHelper;

    @Resource
    private CacheHolder cacheHolder;

    @Resource
    private PushMsgService pushMsgService;

    @Resource
    private ForumProperties forumProperties;

    @Resource
    private UserWarmIntegrationService userWarmIntegrationService;

    @Resource
    private VirtualUserIntegrationService virtualUserIntegrationService;

    @Resource
    private AsyncTaskExecutor taskExecutor;

    @Resource
    private ForumPostInnerService forumPostInnerService;

    @Resource
    private ForumPostTotalFacadeService forumPostTotalFacadeService;

    @Resource
    private UserAuthIntegrationService userAuthIntegrationService;

    @Override
    public PageWarper<NewsReplyDTO> queryReplyList(NewsReplyParam newsReplyParam, BasicParam basicParam) {
        // 考虑到现在评论都很少，采用全部查询的方式代替，如果以后出现超级多评论，那需要重构方案了
        List<NewsReplyDTO> allReplyList = this.newsReplyMapper.queryAllReplyWithContent(newsReplyParam.getNewsId(), newsReplyParam.getUserId());

        // 数据填充，二级评论、额外信息（是否VIP）等
        allReplyList = commentFacadeService.execReplyLogic(allReplyList, newsReplyParam, false);

        boolean isOutWeb = basicParam.getPlatform() == 3;

        // 处理分页逻辑
        int start = newsReplyParam.getPageSize() * (newsReplyParam.getPageNum() - 1);
        int total = allReplyList.size();
        allReplyList = allReplyList.stream()
                .skip(start)
                .limit(newsReplyParam.getPageSize())
                .collect(Collectors.toList());

        // 处理content
        allReplyList.forEach(replyDTO -> {
            processReplyContent(replyDTO, isOutWeb);
            replyDTO.getList().forEach(sub -> {
                processReplyContent(sub, isOutWeb);
            });
        });

        // 为了兼容之前的返回模型，这里返回一个包装类
        PageWarper<NewsReplyDTO> result = new PageWarper<>(allReplyList);
        result.setTotal(total);
        int currentIndex = start + newsReplyParam.getPageSize();
        result.setHasNextPage(currentIndex < total);
        result.setIsLastPage(!result.isHasNextPage());
        return result;
    }

    /**
     * 判断当前用户是否对此回复点赞
     */
    private boolean isLike(Long userId, Long replyId) {
        return null != newsReplyLikeMapper.selectByModel(new NewsReplayLike(replyId, userId));
    }

    @Override
    public Json<ReplyDTO> doReply(NewsReplyAddParam newsReplyAddParam, BasicParam basicParam) {
        if (log.isDebugEnabled()) {
            log.debug("[doReply]发表评论，参数:[{}]", toJSONString(newsReplyAddParam));
        }

        Message message = userAuthIntegrationService.hasCommentAuth(newsReplyAddParam.getUserId());
        if (!message.isSuccess()) {
            return Json.badReqeuset(message.getLastMessage());
        }

        List<String> badWords = sensitiveWordFacadeService.match(newsReplyAddParam.getReplyContent(), false, false);
        if (badWords.size() > 0) {
            return ResultUtil.genFailedResult("您发表的评论中有敏感内容[" + badWords.get(0) + "]，请修改");
        }

        // 对象转换
        NewsReply newsReply = newsReplyConver.convert(newsReplyAddParam);

        // 插入评论数据 news_reply
        this.newsReplyMapper.insertSelective(newsReply);

        //若是帖子，则需要加上评论人是否楼主的判断
        if (ReplyTypeEnum.POST_REPLY.getCode() == newsReply.getType()) {
            Long userId = forumPostInnerService.getPostUserId(newsReply.getNewsId());
            if (null != userId) {
                newsReply.setUserIsAuthor(userId.equals(newsReply.getUserId()));
                newsReply.setParentUserIsAuthor(userId.equals(newsReply.getParentUserId()));
            }
        }
        //保存至用户评论中
        UserReply userReply = new UserReply();
        BeanUtils.copyProperties(newsReply, userReply);

        if (log.isDebugEnabled()) {
            log.debug("用户userId=[{}]发表评论,userReply参数:[{}]", userReply.getUserId(), toJSONString(userReply));
        }

        // user_reply
        this.userReplyMapper.insertSelective(userReply);
        //兼容评论时间
        newsReply.setReplyTime(DateBeautifyUtils.timeConvertString(newsReply.getAddTime()));
        //异步执行
        NewsReply param = new NewsReply();
        BeanUtils.copyProperties(newsReply, param);
        //计算帖子,新闻,小视频的评论数,及帖子的评论概览
        this.updateComment(newsReply);

        //增加用户表评论数统计
        userIntegrationService.addUserStatisticsNum(newsReplyAddParam.getUserId(), 3);

        if (log.isDebugEnabled()) {
            log.debug("=======reply : {} ", param);
        }

        // 异步调用评论后的处理
        taskExecutor.execute(() -> processWhenReply(param, newsReplyAddParam, basicParam.getPlatform(),
                newsReplyAddParam.getAreaCode()));
        //隐藏父级评论
        if (newsReply.getLevel().equals(NewsConstant.REPLY_LEVEL_1)
                && basicParam.getPlatform() != PlatformEnum.APPLET.getCode()) {
            newsReply.setParentUserId(null);
            newsReply.setParentUserNickname(null);
        }

        // 构件返回数据
        ReplyDTO result = new ReplyDTO();
        BeanUtils.copyProperties(newsReply, result);

        result.setReplyContent(newsReply.getReplyContentHtml());

        return Json.build(result);
    }

    private void clearCache(Long newsId) {
        if (null == newsId) {
            return;
        }
        cacheHolder.sendEvictCmd(MemoryCacheKey.HOT_REPLY_CACHE, newsId);
        cacheHolder.sendEvictCmd(MemoryCacheKey.REPLY_COUNT_CACHE, newsId);
    }

    /**
     * 当评论类型是帖子时,保存帖子的评论记录到缓存
     *
     * @param userId
     * @param postId
     */
    @Override
    public void saveForumPostReplyRecord(Long userId, Long postId, Date sendTime) {
        Date expireTime = this.getExpireTime();
        KeyGenerator postKey = RedisConfig.FORUM_REPLY_POSTID.copy();
        redisSetAdapter.add(postKey, postId);
        redisListAdapter.expire(postKey, expireTime);
        ForumReplyInfo info = ForumReplyInfo.builder()
                .userId(userId)
                .replyTime(sendTime)
                .build();
        KeyGenerator infoKey = RedisConfig.FORUM_REPLY_INFO.copy().appendKey(postId);
        redisListAdapter.leftPush(infoKey, info);
        redisListAdapter.expire(infoKey, expireTime);
    }

    /**
     * 获取缓存统计帖子评论的缓存过期时间的过期时间
     *
     * @return
     */
    private Date getExpireTime() {
        //当前时间
        Calendar current = Calendar.getInstance();
        int hour = current.get(Calendar.HOUR_OF_DAY);
        //过期时间
        Calendar expired = Calendar.getInstance();
        expired.set(Calendar.MINUTE, 2);
        expired.set(Calendar.SECOND, 0);
        if (hour < 9) {
            expired.set(Calendar.HOUR_OF_DAY, 9);
        } else if (hour < 12) {
            expired.set(Calendar.HOUR_OF_DAY, 12);
        } else if (hour < 18) {
            expired.set(Calendar.HOUR_OF_DAY, 18);
        } else if (hour < 21) {
            expired.set(Calendar.HOUR_OF_DAY, 21);
        } else {
            expired.add(Calendar.DAY_OF_MONTH, 1);
            expired.set(Calendar.HOUR_OF_DAY, 9);
        }
        return expired.getTime();
    }

    @Async
    @Override
    public void processWhenReply(NewsReply newsReply, NewsReplyAddParam newsReplyAddParam, Integer platform, String areaCode) {
        try {
            // 7. 清除缓存
            clearCache(newsReply.getNewsId());

            // 判断是否需要进行审核 有图片则需要审核
            if (isBlank(newsReply.getReplyImg())) {
                // 不审核就进行推送啥的
                processWhenReplySuccess(newsReply);
            }

        } catch (Exception e) {
            log.error("帖子评论推送异步处理失败 newsReply: {}, newsReplyAddParam: {}, platform: {}, areaCode: {}",
                    JSON.toJSONString(newsReply), JSON.toJSONString(newsReplyAddParam), JSON.toJSONString(platform),
                    JSON.toJSONString(areaCode), e);
        }

    }


    @Override
    public void processWhenReplySuccess(NewsReply newsReply) {
        try {
            ForumPostVo forumPostVo = null;
            //1.新闻不是根级评论时将根级评论互动值+3
            LocationDTO location = locationIntegrationService.getLocationByGeocode(newsReply.getAreaCode());
            if (Objects.nonNull(newsReply.getRootId()) && newsReply.getRootId() != 0) {
                NewsReply rootReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReply.getRootId(), newsReply.getNewsId());
                Integer interactiveCount = rootReply.getInteractiveCount();
                interactiveCount = (Objects.isNull(interactiveCount) || interactiveCount == 0) ? 3 : interactiveCount + 3;
                rootReply.setInteractiveCount(interactiveCount);
                newsReplyMapper.updateByPrimaryKeySelective(rootReply);
                UserReply userRootReply = new UserReply();
                BeanUtils.copyProperties(rootReply, userRootReply);
                userReplyMapper.updateinteractiveCountById(userRootReply);
            }
            //2.推送消息，自己给自己评论不推送信息
            if (Objects.nonNull(newsReply.getParentUserId()) &&
                    !(newsReply.getParentUserId().equals(newsReply.getUserId()))) {

                if (log.isDebugEnabled()) {
                    log.debug("封装的消息格式为：[{}]", toJSONString(newsReply));
                }

                if (newsReply.getType() == ReplyTypeEnum.POST_REPLY.getCode()) {
                    forumPostVo = forumPostInnerService.getBriefInfo(newsReply.getNewsId());
                    FormPostContentUtil.replace(forumPostVo, location);
                    forumPostImageHelper.exchangeDetailPost(forumPostVo);
                    messageService.pushNewReplyMessage(newsReply, PushMessageEnum.POST_REPLY, forumPostVo);
                }
            }
            //3.回复帖子的推送
            if (null == newsReply.getParentUserId() && newsReply.getType() == ReplyTypeEnum.POST_REPLY.getCode()) {
                if (Objects.isNull(forumPostVo)) {
                    forumPostVo = forumPostInnerService.getBriefInfo(newsReply.getNewsId());
                }

                FormPostContentUtil.replace(forumPostVo, location);
                forumPostImageHelper.exchangeDetailPost(forumPostVo);
                messageService.pushPostReplyMessage(newsReply, forumPostVo);
            }

            //4.上报评论信息到用户推荐数据源服务
            UserActionEvent userActionEvent = CommentActionEvent.of()
                    .setLastCommentTime(new Date())
                    .setTargetId(newsReply.getNewsId())
                    .setUserId(newsReply.getUserId());

            userActionEventBus.post(userActionEvent);

            // 5.如果评论内容当中涉及到@用户 则需要进行@用户的处理
            if (isNotBlank(newsReply.getReplyContent()) && newsReply.getReplyContent().contains("@")) {
                pushPostReplyATMessage(newsReply, forumPostVo);
            }

            // 6.如果不是图片评论直接增加温暖值和勋章统计数据
            handOutWarm(newsReply.getUserId(), WarmRuleEnum.COMMENT);

            // 7. 清除缓存
            clearCache(newsReply.getNewsId());

            // 8.将评论数据存入缓存用来发送定时回复提醒任务
            if (ReplyTypeEnum.POST_REPLY.getCode() == newsReply.getType()) {
                this.saveForumPostReplyRecord(newsReply.getUserId(), newsReply.getNewsId(), new Date());
            }

        } catch (Exception e) {
            log.error("帖子评论推送异步处理失败 newsReply: {}",
                    JSON.toJSONString(newsReply), e);
        }

    }

    @Override
    public void pushPostReplyATMessage(NewsReply newsReply, ForumPostVo forumPostVo) {
        if (Objects.isNull(forumPostVo)) {
            forumPostVo = forumPostInnerService.getBriefInfo(newsReply.getNewsId());
        }

        // 提取HTTP 拼接一下 默认的是没有 p标签的
        Document document = Jsoup.parse("<p>" + newsReply.getReplyContentHtml() + "</p>");
        List<Long> userIds = AtUserIdExtractUtils.processAtInfo(document);

        // 去个重 避免多次发送
        userIds = Lists.newArrayList(Sets.newHashSet(userIds));

        if (log.isDebugEnabled()) {
            log.debug("帖子评论: {} 中包含@用户数据，给用户: {} 推送", newsReply.getReplyContentHtml(), userIds);
        }

        // @用户推送
        messageService.pushPostReplyATMessage(newsReply, forumPostVo, userIds);
    }

    /**
     * 监听需要@推送的事件
     *
     * @param event 事件
     */
    @EventListener(ReplyAccessApplicationEvent.class)
    public void onReplyAccessApplicationEvent(ReplyAccessApplicationEvent event) {

        try {
            if (log.isDebugEnabled()) {
                log.debug("接收到评论审核通过的事件，event: {}", JSON.toJSONString(event));
            }

            NewsReply newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(event.getReplyId(), event.getNewsId());

            if (Objects.isNull(newsReply)) {
                log.warn("event: {} 无法获取到评论信息， 无法进行推送", JSON.toJSONString(event));
                return;
            }

            // 异步调用
            taskExecutor.execute(() -> processWhenReplySuccess(newsReply));
        } catch (Exception e) {
            log.error("接收到评论审核通过的事件出现错误，event: {}", JSON.toJSONString(event), e);
        }
    }

    @Override
    public void updateComment(NewsReply newsReply) {
        int count = newsReplyMapper.selectCountByNewsId(newsReply.getNewsId());
        if (ReplyTypeEnum.POST_REPLY.getCode() == newsReply.getType()) {
            //维护帖子评论概览及帖子的评论数
            ForumPostStatistic forumPostStatistic = ForumPostStatistic.buildComments(newsReply.getNewsId(), count);
            List<NewsReply> replyList = newsReplyMapper.selectPostReplyByNewsIdOrderByAddTime(newsReply.getNewsId());
            if (CollectionUtils.isNotEmpty(replyList)) {
                forumPostStatistic.setReplyInfo(toJSONString(replyList));
            }

            ForumPostTotalEntity entity = ForumPostTotalEntity.builder()
                    .commentCount(count)
                    .postId(newsReply.getNewsId())
                    .build();
            forumPostTotalFacadeService.updateField(entity);
        }
    }

    @Override
    public PageWarper<MyReplysVO> selectMyReplys(UserReplyParam newsReplyParam, BasicParam basicParam) {
        PageWarper<MyReplysVO> myReplysVOPageWarper = new PageWarper<>(this.userReplyMapper.selectMyReplys(newsReplyParam));
        List<MyReplysVO> myReplysVOList = myReplysVOPageWarper.getList();
        LocationDTO location = locationIntegrationService.getLocationByGeocode(newsReplyParam.getAreaCode());

        boolean isOutWeb = basicParam.getPlatform() == 3;

        for (MyReplysVO myReplysVO : myReplysVOList) {
            Date addTime = myReplysVO.getAddTime();
            String replyTime = DateBeautifyUtils.timeConvertString(addTime);
            myReplysVO.setReplyTime(replyTime);
            Byte type = myReplysVO.getType();

            if (type == ReplyTypeEnum.POST_REPLY.getCode()) {
                ForumPostVo forumPostVo = forumPostInnerService.getBriefInfo(myReplysVO.getNewsId());
                if (null != forumPostVo) {
                    //查看帖子是否开启占位符
                    FormPostContentUtil.replace(forumPostVo, location);
                    //封面图处理
                    forumPostImageHelper.exchangeDetailPost(forumPostVo);
                    ReplyPostDTO replyPostDTO = getForumPostReplyDto(forumPostVo);
                    myReplysVO.setReplyPostDto(replyPostDTO);
                    myReplysVO.setTitle(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()));
                }
            }

            // 处理帖子评论
            processReplyContent(myReplysVO, isOutWeb);
        }
        return myReplysVOPageWarper;
    }

    @Override
    public NewsReplyDTO delMyReply(Long replyId, Long userId, Long newsId, Byte isUserDelete) {
        NewsReply newsReply = null;
        if (null != newsId && 0 != newsId) {
            newsReply = this.newsReplyMapper.selectByPrimaryKeyAndNewsId(replyId, newsId);
        }
        if (Objects.isNull(newsReply)) {
            newsReply = this.newsReplyMapper.selectByPrimaryKey(replyId);
        }
        //获取评论原来的状态
        Byte beforeReplyStatus = newsReply.getStatus();

        //删除评论逻辑
        newsReply.setIsUserDelete(isUserDelete);
        this.replyRelationship(newsReply);
        NewsReplyDTO newsReplyDTO = new NewsReplyDTO();
        BeanUtils.copyProperties(newsReply, newsReplyDTO);
        //评论状态置为删除
        newsReplyDTO.setDeleteFlag((byte) 1);
        newsReplyDTO.setReplyTime(DateBeautifyUtils.timeConvertString(newsReplyDTO.getAddTime()));
        newsReplyDTO.setReplyContent("该评论已删除！");
        newsReplyDTO.setIsLike(isLike(userId, replyId) ? 1 : 0);
        if (newsReply.getRootId() == 0) {
            List<NewsReplyMirrorDTO> replyMirrorDTOS = newsReplyMapper.selectSubReplyList(replyId, newsReply.getNewsId(), null);
            List<Long> replyLike = newsReplyMapper.selectReplyLike(newsReply.getNewsId(), userId, replyId);
            replyMirrorDTOS.forEach(e -> {
                commentFacadeService.dealSubMirrorDTO(e, userId, replyLike);
            });
            newsReplyDTO.setList(replyMirrorDTOS);
        }
        if (newsReply.getLevel() == 1) {
            newsReplyDTO.setParentUserId(null);
            newsReplyDTO.setParentUserNickname(null);
        }
        //用户表评论数统计-1
        userIntegrationService.addUserStatisticsNum(userId, 4);
        //维护新闻,帖子,小视频评论数
        this.updateComment(newsReply);

        clearCache(newsId);

        // 非用户删除
        if (!Objects.equals(isUserDelete, (byte) 1)) {
            // 推送给用户 告知评论被删除
            pushMsgService.pushReplyDeleteAndRejectMsg(newsReply.getReplyContent(),
                    newsReply.getUserId(),
                    forumProperties.getOfficialRulePostId());
        }

        //判断如果之前的状态是正常展示，则删除之后扣除温暖值和成就勋章统计值
        if (Objects.equals(ReplyStatusEnum.IS_SHOW.getCode(), beforeReplyStatus)) {
            handOutWarm(userId, Objects.equals(isUserDelete, (byte) 1) ? WarmRuleEnum.ONESELF_DEL_COMMENT : WarmRuleEnum.OPERATE_DEL_COMMENT);
        }

        return newsReplyDTO;
    }


    private void handOutWarm(Long userId, WarmRuleEnum warmRuleEnum) {
        userWarmIntegrationService.updateWarm(UserWarmActionParam.builder()
                .warmRuleEnum(warmRuleEnum)
                .userId(userId)
                .build());
    }

    /**
     * 评论删除时处理评论与回复
     *
     * @param newsReply
     */
    private void replyRelationship(NewsReply newsReply) {
        //当删除的评论时根评论时
        this.userReplyMapper.deleteByPrimaryKey(newsReply.getId(), newsReply.getUserId(), newsReply.getIsUserDelete());
        if (newsReply.getRootId() == 0) {
            List<NewsReplyMirrorDTO> list = newsReplyMapper.selectSubReplyList(newsReply.getId(), newsReply.getNewsId(), null);
            if (CollectionUtils.isNotEmpty(list)) {
                newsReply.setDeleteFlag((byte) 2);
                newsReplyMapper.updateByPrimaryKeySelective(newsReply);
            } else {
                this.newsReplyMapper.deleteByPrimaryKey(newsReply.getId(), newsReply.getNewsId(), newsReply.getIsUserDelete());
            }
        } else {
            //删除评论不为根评论时
            this.newsReplyMapper.deleteByPrimaryKey(newsReply.getId(), newsReply.getNewsId(), newsReply.getIsUserDelete());
            List<NewsReplyMirrorDTO> list = newsReplyMapper.selectSubReplyList(newsReply.getRootId(), newsReply.getNewsId(), null);
            NewsReply rootReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReply.getRootId(), newsReply.getNewsId());
            if (rootReply.getDeleteFlag() != 0) {
                if (CollectionUtils.isNotEmpty(list)) {
                    NewsReply record = new NewsReply();
                    record.setId(newsReply.getRootId());
                    record.setNewsId(newsReply.getNewsId());
                    record.setDeleteFlag((byte) 2);
                    newsReplyMapper.updateByPrimaryKeySelective(record);
                }
            }

        }
    }


    @Async
    @Override
    public void doTriggerUpdateInfo(Long userId, String nickname, String headImg) {
        //更新自己的评论
        List<NewsReply> newsReplyList = newsReplyMapper.selectByUser(userId);
        for (NewsReply newsReply : newsReplyList) {
            NewsReply newsReplyUp = new NewsReply();
            newsReplyUp.setId(newsReply.getId());
            newsReplyUp.setUserId(userId);
            newsReplyUp.setNewsId(newsReply.getNewsId());
            newsReplyUp.setUserNickname(nickname);
            newsReplyUp.setHeadImg(headImg);

            UserReply userReply = new UserReply();
            BeanUtils.copyProperties(newsReplyUp, userReply);

            newsReplyMapper.updateUserInfo(newsReplyUp);
            userReplyMapper.updateUserInfo(userReply);

            clearCache(newsReply.getNewsId());
        }
        //更新父级的评论
        List<NewsReply> parentNewsReplyList = newsReplyMapper.selectByParentUser(userId);
        for (NewsReply newsReply : parentNewsReplyList) {
            NewsReply newsReplyUp = new NewsReply();
            newsReplyUp.setId(newsReply.getId());
            newsReplyUp.setUserId(userId);
            newsReplyUp.setNewsId(newsReply.getNewsId());
            newsReplyUp.setParentUserNickname(nickname);
            newsReplyUp.setParentHeadImg(headImg);

            UserReply userReply = new UserReply();
            BeanUtils.copyProperties(newsReplyUp, userReply);

            newsReplyMapper.updateParentUserInfo(newsReplyUp);
            userReplyMapper.updateParentUserInfo(userReply);

            clearCache(newsReply.getNewsId());
        }

    }

    @Override
    public Message doProduceNewsReplyLike(NewsReplyLikeParam newsReplyLikeParam) {
        if (!checkLikeParam(newsReplyLikeParam)) {
            return Message.build(false, "参数验证错误");
        }
        if (log.isDebugEnabled()) {
            log.debug("点赞的参数是:{}", toJSONString(newsReplyLikeParam));
        }

        KeyGenerator keyGenerator = RedisConfig.NEWS_QUEUE.copy().setKey("newsReplyLikeQueue");

        redisListAdapter.leftPush(keyGenerator, newsReplyLikeParam);
        this.callAsyncConsume();
        return Message.build(true);
    }

    private void callAsyncConsume() {
        NewNewsReplyService newsReplyService = SpringContextHolder.getBean(NewNewsReplyService.class);
        newsReplyService.doNewsReplyLikeConsume();
    }

    @Async
    @Override
    public void doNewsReplyLikeConsume() {
        KeyGenerator keyGenerator = RedisConfig.NEWS_QUEUE.copy().setKey("newsReplyLikeQueue");

        NewsReplyLikeParam newsReplyLikeWarper = redisListAdapter.rightPop(keyGenerator, NewsReplyLikeParam.class);
        if (null != newsReplyLikeWarper) {
            if (log.isDebugEnabled()) {
                log.debug("新闻回复点赞开始消费...");
            }
            int type = newsReplyLikeWarper.getType();
            NewsReply newsReply;
            if (null != newsReplyLikeWarper.getNewsId() && 0 != newsReplyLikeWarper.getNewsId()) {
                newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReplyLikeWarper.getReplyId(), newsReplyLikeWarper.getNewsId());
            } else {
                newsReply = newsReplyMapper.selectByPrimaryKey(newsReplyLikeWarper.getReplyId());
            }
            if (newsReply != null) {
                int count = saveReplyLiKeRecord(newsReply.getId(), newsReplyLikeWarper.getUserId(), type);
                if (count > 0 && type == 1) {
                    //1.增加评论点赞记录
                    if (log.isDebugEnabled()) {
                        log.debug("评论实体:{}", JSONObject.toJSON(newsReply));
                    }

                    int likeCount = generatorLikeCount(type, newsReply.getLikeCount());
                    if (log.isDebugEnabled()) {
                        log.debug("点赞数:{}", likeCount);
                    }

                    NewsReply newsReplyUp = new NewsReply();
                    newsReplyUp.setId(newsReplyLikeWarper.getReplyId());
                    newsReplyUp.setLikeCount(likeCount);
                    newsReplyUp.setUserId(newsReply.getUserId());
                    newsReplyUp.setNewsId(newsReply.getNewsId());
                    //根评论才计算互动值
                    if (newsReply.getRootId() == 0) {
                        newsReplyUp.setInteractiveCount(newsReply.getInteractiveCount() + 1);
                    }
                    newsReplyMapper.updateByPrimaryKeySelective(newsReplyUp);
                    UserReply userReply = new UserReply();
                    BeanUtils.copyProperties(newsReplyUp, userReply);
                    userReplyMapper.updateinteractiveCountById(userReply);

                    //2.评论点赞推送
                    if ((!newsReplyLikeWarper.getUserId().equals(newsReply.getUserId())) && newsReplyLikeWarper.getType() == 1) {
                        LocationDTO location = locationIntegrationService.getLocationByGeocode(newsReplyLikeWarper.getAreaCode());
                        if (newsReply.getType() == 3) {
                            ForumPostVo postVo = forumPostInnerService.getBriefInfo(newsReply.getNewsId());
                            FormPostContentUtil.replace(postVo, location);
                            forumPostImageHelper.exchangeDetailPost(postVo);
                            messageService.pushNewLikeMessage(newsReplyLikeWarper, PushMessageEnum.POST_LIKE, newsReply, postVo);
                        }
                    }

                    //调用增加用户点赞数接口
                    userIntegrationService.updateUserLikeNumByUserId(newsReply.getUserId());

                    // 判断是否是热评
                    if (log.isDebugEnabled()) {
                        log.debug("评论: {} 是否是热评: {}", newsReply.getId(), newsReply.getHotReply());
                    }

                    if (Objects.equals(newsReply.getHotReply(), 1)) {
                        //  刷新热评缓存
                        cacheHolder.sendEvictCmd(MemoryCacheKey.HOT_REPLY_CACHE, Objects.toString(newsReply.getNewsId()));
                    }
                }
            }
            doNewsReplyLikeConsume();
        }
    }

    @Override
    public NewsReplyDetailDTO getNewsReplyDetailDTO(Long replyId, Long userId, Long newsId, String areaCode, BasicParam basicParam) {
        NewsReply newsReply;
        if (null != newsId && 0 != newsId) {
            newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(replyId, newsId);
        } else {
            newsReply = newsReplyMapper.selectByPrimaryKey(replyId);
        }
        if (Objects.isNull(newsReply) || Objects.isNull(newsReply.getRootId())) {
            return null;
        }
        //判断评论是否是跟评论,不是则找出根评论
        if (newsReply.getRootId() != 0) {
            if (null != newsId && 0 != newsId) {
                newsReply = newsReplyMapper.selectByPrimaryKeyAndNewsId(newsReply.getRootId(), newsId);
            } else {
                newsReply = newsReplyMapper.selectByPrimaryKey(newsReply.getRootId());
            }
        }
        NewsReplyDetailDTO newsReplyDetailDTO = new NewsReplyDetailDTO();
        if (null != newsReply) {

            BeanUtils.copyProperties(newsReply, newsReplyDetailDTO);

            if (isLike(userId, newsReplyDetailDTO.getId())) {
                newsReplyDetailDTO.setIsLike(1);
            }
            newsReplyDetailDTO.setReplyTime(DateBeautifyUtils.timeConvertString(newsReplyDetailDTO.getAddTime()));
            //添加评论的新闻.帖子.或视频的详细实体
            this.addSuperiorObject(newsReply, userId, newsReplyDetailDTO, areaCode);
            //查询该评论下的回复
            List<NewsReplyMirrorDTO> replies = this.newsReplyMapper.
                    selectSubReplyList(newsReply.getId(), newsReply.getNewsId(), userId);
            if (CollectionUtils.isNotEmpty(replies)) {
                replies.forEach(replay -> {
                    List<Long> replyLike = newsReplyMapper.selectReplyLike(replay.getNewsId(), userId, replyId);
                    commentFacadeService.dealSubMirrorDTO(replay, userId, replyLike);
                });
            }

            replies = (List<NewsReplyMirrorDTO>) filter(replies, userId);

            newsReplyDetailDTO.setList(replies);

            boolean isOutWeb = basicParam.getPlatform() == 3;
            // 处理@和图片
            processReplyContent(newsReplyDetailDTO, isOutWeb);

            // 子评论
            replies.forEach(reply -> {
                // 处理@和图片
                processReplyContent(reply, isOutWeb);
            });
        }
        return newsReplyDetailDTO;
    }

    /**
     * 添加评论新闻.帖子.或视频的详细实体
     *
     * @param newsReply
     * @param userId
     * @param newsReplyDetailDTO
     * @return
     */
    private NewsReplyDetailDTO addSuperiorObject(NewsReply newsReply, Long userId, NewsReplyDetailDTO newsReplyDetailDTO, String areaCode) {
        Byte type = newsReply.getType();
        LocationDTO location = locationIntegrationService.getLocationByGeocode(areaCode);
        if (type == ReplyTypeEnum.POST_REPLY.getCode()) {
            ForumPostVo forumPostVo = forumPostInnerService.getBriefInfo(newsReply.getNewsId());
            FormPostContentUtil.replace(forumPostVo, location);
            if (null != forumPostVo) {
                ReplyPostDTO replyPostDTO = getForumPostReplyDto(forumPostVo);
                newsReplyDetailDTO.setReplyPostDto(replyPostDTO);
                newsReplyDetailDTO.setTitle(ExtendFactory.getTitle(forumPostVo.getTitle(), forumPostVo.getTextField()));
            }
        }
        return newsReplyDetailDTO;
    }

    private ReplyPostDTO getForumPostReplyDto(ForumPostVo forumPostVo) {
        ReplyPostDTO replyPostDTO = new ReplyPostDTO();
        replyPostDTO.setId(forumPostVo.getId());
        replyPostDTO.setStatus((byte) forumPostVo.getStatus().intValue());
        replyPostDTO.setTitle(forumPostVo.getTitle());
        replyPostDTO.setPostImgList(forumPostVo.getPostImgList());

        return replyPostDTO;
    }

    /**
     * 得到更新之后的点赞数
     *
     * @param type
     * @param likeCount
     * @return
     */
    private int generatorLikeCount(int type, int likeCount) {
        if (0 == type) {
            if (likeCount > 0) {
                likeCount -= 1;
            }
        } else {
            likeCount += 1;
        }
        return likeCount;
    }

    private int saveReplyLiKeRecord(Long replyId, Long userId, int type) {
        int count = 0;
        NewsReplayLike newsReplayLike = newsReplyLikeMapper.selectByModel(new NewsReplayLike(replyId, userId));
        if (type == 1) {
            if (null == newsReplayLike) {
                NewsReplayLike newsReplayLikeNew = new NewsReplayLike();
                newsReplayLikeNew.setAddTime(new Date());
                newsReplayLikeNew.setId(SequenceHolder.nextLongId());
                newsReplayLikeNew.setReplyId(replyId);
                newsReplayLikeNew.setUserId(userId);
                count = newsReplyLikeMapper.insert(newsReplayLikeNew);

            }
        } else {
            if (null != newsReplayLike) {
                count = newsReplyLikeMapper.deleteByPrimaryKey(newsReplayLike.getId());
            }
        }
        return count;
    }


    /**
     * 评论点赞时验证参数
     *
     * @param newsReplyLikeParam
     * @return
     */
    private boolean checkLikeParam(NewsReplyLikeParam newsReplyLikeParam) {
        return null != newsReplyLikeParam.getReplyId() && null != newsReplyLikeParam.getUserId();
    }


    @Override
    public Boolean doDealHistoryReply() {
        //删除新闻评论表中user_id = 0,parent_user_id = 0 数据
        newsReplyMapper.deleteNewsReply();
        List<VirtualUser> userList = virtualUserIntegrationService.getRandom(1000);
        for (int i = 0; i < 10; i++) {
            String tableName = "t_news_reply_" + i;
            List<NewsReply> replyList = newsReplyMapper.selectByUserIdIsZero(tableName);
            dealReplyList(replyList, userList);
        }
        return Boolean.TRUE;
    }

    @Override
    public void statisticalPostReply() {
        //获得一段时间内,所有的帖子评论数量
        KeyGenerator postIdKey = RedisConfig.FORUM_REPLY_POSTID.copy();
        try {
            //开启线程的数量
            int threadNum = 10;
            Set<Long> postIds = redisSetAdapter.getAllMembers(postIdKey, new TypeReference<Long>() {
            });
            CountDownLatch countDownLatch = new CountDownLatch(threadNum);
            for (int i = 0; i < threadNum; i++) {
                final int j = i;
                List<Long> collect = postIds.stream()
                        .filter(e -> e % threadNum == j)
                        .collect(Collectors.toList());
                asyncTaskExecutor.execute(new PostReplyThread(collect, countDownLatch));
            }
            countDownLatch.await();
        } catch (InterruptedException e) {
            log.error("线程等待异常", e);
        } finally {
            redisSetAdapter.remove(postIdKey);
        }
    }


    @Override
    public NewsReplyListDTO replyList(NewsReplyParam param, BasicParam basicParam) {
        NewsReplyListDTO result = new NewsReplyListDTO();

        //热门评论
        result.setHotReplys(commentFacadeService.getHotReplays(param));
        // 数据填充，二级评论、额外信息（是否VIP）等
        result.setHotReplys(commentFacadeService.execReplyLogic(result.getHotReplys(), param, true));

        // 获取普通评论
        PageWarper<NewsReplyDTO> newsReplyDTOPageWarper = queryReplyList(param, basicParam);
        //普通评论
        result.setReplys(newsReplyDTOPageWarper);
        boolean isOutWeb = basicParam.getPlatform() == 3;

        // 处理热评内容 @用户的数据
        result.getHotReplys().forEach(hotReply -> {
            processReplyContent(hotReply, isOutWeb);
            if (!org.springframework.util.CollectionUtils.isEmpty(hotReply.getList())) {
                hotReply.getList().forEach(hotSubReply -> processReplyContent(hotSubReply, isOutWeb));
            }
        });


        return result;
    }


    private void dealReplyList(List<NewsReply> replyList, List<VirtualUser> userList) {
        replyList.forEach(e -> {
            int random = ThreadLocalRandom.current().nextInt(0, userList.size());
            VirtualUser userInfo = userList.get(random);
            e.setUserId(userInfo.getId());
            e.setHeadImg(userInfo.getHeadImg());
            e.setUserNickname(userInfo.getNickname());
            UserReply userReply = new UserReply();
            BeanUtils.copyProperties(e, userReply);
            newsReplyMapper.updateReplyUserInfo(e);
            userReplyMapper.insertSelective(userReply);

        });
    }

    @Override
    public void delReply(Long id, Long newsId) {
        AdminNewsReply adminNewsReply = adminNewsReplyMapper.selectByPrimaryKey(id, newsId);
        if (adminNewsReply != null) {
            //调用新闻服务删除评论接口
            delMyReply(adminNewsReply.getId(), adminNewsReply.getUserId(), adminNewsReply.getNewsId(), (byte) 0);

            //发送评论消息召回
            messageService.sendMessageFilterForReply(null, id);
        }
    }


}
