package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.news.config.LocalCrumbsProperties;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.facade.service.ForumTopicFacadeService;
import com.bxm.localnews.news.model.vo.LocalCrumbs;
import com.bxm.localnews.news.model.vo.topic.TopicCrumbsExtData;
import com.bxm.localnews.news.model.vo.topic.TopicVo;
import com.bxm.localnews.news.service.LocalCircleService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;

/**
 * 本地圈service实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2020/9/22 2:08 下午
 */
@Service
public class LocalCircleServiceImpl implements LocalCircleService {

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private ForumTopicFacadeService forumTopicFacadeService;

    @Resource
    private LocalCrumbsProperties localCrumbsProperties;

    @Override
    public LocalCrumbs getLocalCrumbsInfo(String userId, String areaCode) {
        //如果areaCode为空则返回一些默认的空数据
        if (StringUtils.isBlank(areaCode)) {
            return getEmptyResult();
        }


        LocalCrumbs topicSquareCrumbs = new LocalCrumbs();


        //1、从Apollo中获取，面包块的名称，设置面包块的类型,设置面包块背景图片
        topicSquareCrumbs.setName(localCrumbsProperties.getTopicSquareTitle());
        topicSquareCrumbs.setType(1);
        // 设置面包块背景图片
        topicSquareCrumbs.setBackgroundImgUrl(localCrumbsProperties.getTopicSquareImgUrl());
        // 设置面包块跳转地址
        topicSquareCrumbs.setJumpUrl(localCrumbsProperties.getTopicSquareJumpUrl());

        //2、获取当前areaCode对应的城市有多少条帖子
        //   获取当前areaCode对应的城市有多少小纸条

        //全国贴总数redis Key
        KeyGenerator nationalCrumbsTotal = getNationalAreaTotal();

        KeyGenerator topicSquareAreaTotal = getCrumbsAreaTotalKey(1);
        KeyGenerator topicSquareReadCount = getCrumbsReadCount(1, areaCode);

        // 新增帖子总数：包含地区贴数量 + 全国贴数量
        long postTotal = 0;

        //获取区域帖子总数
        Long areaPostTotal = redisHashMapAdapter.get(topicSquareAreaTotal, areaCode, Long.class);

        // 获取全国贴总数
        Long nationalPostTotal = redisHashMapAdapter.get(nationalCrumbsTotal, "1", Long.class);

        if (nationalPostTotal != null) {
            postTotal += nationalPostTotal;
        }
        if (areaPostTotal != null) {
            postTotal += areaPostTotal;
        }

        // 如果键不存在,初始化为0
        if (Boolean.FALSE.equals(redisHashMapAdapter.exists(topicSquareReadCount, userId))) {
            redisHashMapAdapter.put(topicSquareReadCount, userId, 0);
        }

        //3、获取当前用户在这个areaCode下上次，已读多少个帖子
        //   获取当前用户在这个areaCode下上次，已读多少个纸条
        Long currentUserReadPostCount = redisHashMapAdapter.get(topicSquareReadCount, userId, Long.class);

        //4、用areaCode下有多少帖子 - 当前用户已读帖子 = 显示新增帖子数
        //   用areaCode下有多少小纸条 - 当前用户已读小纸条 = 显示新增小纸条数
        if (postTotal > 0 && postTotal > currentUserReadPostCount) {
            topicSquareCrumbs.setLatestCount(postTotal - currentUserReadPostCount);
        }
        //5、 取最新上线的话题进行显示
        TopicVo topicLatest = forumTopicFacadeService.getTopicLatest(areaCode);

        TopicCrumbsExtData topicCrumbsExtData = new TopicCrumbsExtData();

        // 设置扩展数据
        topicSquareCrumbs.setExtData(topicCrumbsExtData);

        topicCrumbsExtData.setLatestTopicId(topicLatest.getId());
        topicCrumbsExtData.setLatestTopicTitle("#" + topicLatest.getTitle() + "#");

        return topicSquareCrumbs;
    }

    /**
     * 返回一个空的列表
     *
     * @return 本地圈面包块展示数据
     */
    private LocalCrumbs getEmptyResult() {
        TopicCrumbsExtData topicCrumbsExtData = new TopicCrumbsExtData();
        // 设置扩展数据
        topicCrumbsExtData.setLatestTopicId(null);
        topicCrumbsExtData.setLatestTopicTitle("");

        LocalCrumbs topicSquareCrumbs = new LocalCrumbs();
        topicSquareCrumbs.setName(localCrumbsProperties.getTopicSquareTitle());
        topicSquareCrumbs.setLatestCount(0L);
        topicSquareCrumbs.setType(1);
        topicSquareCrumbs.setJumpUrl(localCrumbsProperties.getTopicSquareJumpUrl());
        topicSquareCrumbs.setBackgroundImgUrl(localCrumbsProperties.getTopicSquareImgUrl());
        topicSquareCrumbs.setExtData(topicCrumbsExtData);

        return topicSquareCrumbs;
    }

    @Override
    public Boolean reportCrumbsInfo(Long userId, String areaCode, Integer type) {
        // 如果areaCode为空上报数据直接返回成功，因为areaCode为空默认的未读消息也是空，是一种错误情况
        if (StringUtils.isBlank(areaCode)) {
            return Boolean.TRUE;
        }
        //生成指定面包块类型下，对应区域的key
        KeyGenerator crumbsAreaTotal = getCrumbsAreaTotalKey(type);
        KeyGenerator crumbsNationalTotal = getNationalAreaTotal();

        //指定面包块
        KeyGenerator crumbsReadCount = getCrumbsReadCount(type, areaCode);
        // 1、上报面包块的动作，每次将用户已读的帖子更新为当前地区帖子总数
        if (type == 1) {
            long postTotal = 0;

            // 地区帖子总数
            Long areaPostTotal = redisHashMapAdapter.get(crumbsAreaTotal, areaCode, Long.class);
            Long nationalPostTotal = redisHashMapAdapter.get(crumbsNationalTotal, String.valueOf(type), Long.class);

            if (areaPostTotal != null) {
                postTotal += areaPostTotal;
            }

            if (nationalPostTotal != null) {
                postTotal += nationalPostTotal;
            }

            if (postTotal > 0) {
                // 更新用户已读消息数
                redisHashMapAdapter.put(crumbsReadCount, String.valueOf(userId), postTotal);
            }
        }
        return Boolean.TRUE;
    }

    /**
     * 获取面包块新增消息数量缓存key
     *
     * @param type 面包块类型： 1：话题广场，2：同城交友
     * @return 缓存key
     */
    private KeyGenerator getCrumbsAreaTotalKey(int type) {
        return RedisConfig.CRUMBS_AREA_TOTAL.copy().appendKey(type);
    }

    /**
     * 获取用户已读面包块消息数量缓存数量
     *
     * @param type     面包块类型： 1：话题广场，2：同城交友
     * @param areaCode 区域编码
     * @return 缓存key
     */
    private KeyGenerator getCrumbsReadCount(int type, String areaCode) {
        return RedisConfig.CRUMBS_READ_COUNT.copy().appendKey(type).appendKey(areaCode);
    }

    /**
     * 获取全国贴和小纸条新增消息数量缓存key
     *
     * @return 缓存key
     */
    private KeyGenerator getNationalAreaTotal() {
        return RedisConfig.NATIONAL_AREA_TOTAL.copy();
    }
}