package com.bxm.localnews.news.service.impl;

import com.bxm.localnews.dto.UserRegisterInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.config.PlusEntryProperties;
import com.bxm.localnews.news.domain.NewsPublishEntryExtendMapper;
import com.bxm.localnews.news.model.dto.IndexPlusDTO;
import com.bxm.localnews.news.model.dto.IndexPlusUserInfoDTO;
import com.bxm.localnews.news.model.dto.LinkDTO;
import com.bxm.localnews.news.model.entity.NewsPublishEntry;
import com.bxm.localnews.news.model.entity.NewsPublishEntryArea;
import com.bxm.localnews.news.model.param.IndexPlusParam;
import com.bxm.localnews.news.service.PublishEntryService;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.google.common.collect.Maps;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSONString;
import static com.bxm.localnews.news.constant.EntryLevel.MAIN;
import static com.bxm.localnews.news.constant.EntryLevel.SECOND;
import static com.bxm.localnews.news.constant.GlobalPublishType.NO_PUBLISH;
import static com.bxm.localnews.news.constant.GlobalStatus.GLOBAL;
import static com.bxm.localnews.news.constant.GlobalStatus.REGIONAL;
import static com.bxm.localnews.news.constant.RedisCacheKey.PUBLISH_ENTRY_KEY;
import static com.bxm.newidea.component.tools.StringUtils.isGrateOrEqualThan;
import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author gonzo
 * @date 2020-08-10 14:32
 **/
@Slf4j
@Service
@AllArgsConstructor
public class PublishEntryServiceImpl implements PublishEntryService {

    private final NewsPublishEntryExtendMapper newsPublishEntryExtendMapper;

    private final UserIntegrationService userIntegrationService;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final PlusEntryProperties plusEntryProperties;

    @Override
    public IndexPlusDTO index(IndexPlusParam param) {

        if (StringUtils.isBlank(param.getAreaCode())) {
            log.warn("获取发帖入口的快捷入口配置失败，缺少areaCode, 请求参数: {}", toJSONString(param));
            IndexPlusDTO indexPlus = new IndexPlusDTO();
            indexPlus.setPrimary(Collections.emptyList());
            indexPlus.setSecondary(Collections.emptyList());
            return indexPlus;
        }

        // 优先缓存获取
        IndexPlusDTO indexPlus = redisHashMapAdapter.get(PUBLISH_ENTRY_KEY, param.getAreaCode(), IndexPlusDTO.class);

        if (Objects.isNull(indexPlus)
                || indexPlus.getV() == null
                || indexPlus.getV() < IndexPlusDTO.CURRENT_VERSION) {
            // 初始化
            indexPlus = new IndexPlusDTO();
            indexPlus.setPrimary(Collections.emptyList());
            indexPlus.setSecondary(Collections.emptyList());

            // 查询所有的入口数据
            List<NewsPublishEntry> newsPublishEntries = newsPublishEntryExtendMapper.selectAllUpEntry();
            if (CollectionUtils.isEmpty(newsPublishEntries)) {
                // 没有配置数据直接返回
                return indexPlus;
            }

            Map<Byte, Map<Long, List<NewsPublishEntryArea>>> areaInfosTemp = Maps.newHashMap();
            // 查询当前区域的入口配置id
            List<NewsPublishEntryArea> newsPublishEntryAreas = newsPublishEntryExtendMapper.listEntryByArea(param.getAreaCode());
            // 区域配置不为空则进行区域的过滤
            if (!CollectionUtils.isEmpty(newsPublishEntryAreas)) {
                // 根据类型分组
                areaInfosTemp = newsPublishEntryAreas.stream()
                        .collect(Collectors.groupingBy(NewsPublishEntryArea::getType, Collectors.groupingBy(NewsPublishEntryArea::getPublishEntryId)));
            }

            Map<Byte, Map<Long, List<NewsPublishEntryArea>>> areaInfos = areaInfosTemp;
            // 过滤
            newsPublishEntries = newsPublishEntries.stream()
                    .filter(p -> {
                        // 如果当前入口配置的是全国
                        if (Objects.equals(p.getGlobalStatus(), GLOBAL)) {
                            // 不投放的区域为空
                            return Objects.isNull(areaInfos.get(NO_PUBLISH))
                                    // 不投放的区域不为空 但是不包含当前配置
                                    || CollectionUtils.isEmpty(areaInfos.get(NO_PUBLISH).get(p.getId()));

                        } else {
                            // 投放区域不为空
                            return Objects.nonNull(areaInfos.get(REGIONAL))
                                    // 且投放区域包含当前区域
                                    && !CollectionUtils.isEmpty(areaInfos.get(REGIONAL).get(p.getId()));
                        }
                    }).collect(Collectors.toList());


            // 根据入口类型分组
            Map<Byte, List<LinkDTO>> publishMap = newsPublishEntries.stream()
                    .collect(Collectors.groupingBy(NewsPublishEntry::getLevel,
                            Collectors.mapping(this::convert, Collectors.toList())));

            if (!CollectionUtils.isEmpty(publishMap.get(MAIN))) {
                indexPlus.setPrimary(publishMap.get(MAIN));
            }
            if (!CollectionUtils.isEmpty(publishMap.get(SECOND))) {
                indexPlus.setSecondary(publishMap.get(SECOND));
            }
            // 添加缓存
            redisHashMapAdapter.put(PUBLISH_ENTRY_KEY, param.getAreaCode(), indexPlus);
        }

        IndexPlusDTO indexPlusDTO = fillGuideInfo(indexPlus, param);

        //过滤版本
        versionFilter(indexPlusDTO, param);

        return indexPlusDTO;
    }

    /**
     * 根据版本对一级和二级入口进行过滤
     *
     * @param indexPlusDTO 最终返回入口数据
     * @param param        入参
     */
    private void versionFilter(IndexPlusDTO indexPlusDTO, IndexPlusParam param) {
        //一级入口列表
        List<LinkDTO> primaryList = indexPlusDTO.getPrimary();
        //二级入口列表
        List<LinkDTO> secondaryList = indexPlusDTO.getSecondary();

        //过滤一级入口的生效版本
        if (CollectionUtils.isNotEmpty(primaryList)) {
            List<LinkDTO> resultPrimaryList = primaryList.stream().filter(primaryItem -> {
                //如果不做版本限定或者当前版本大于等于生效版本
                return isBlank(primaryItem.getVersion()) || Objects.equals("0", primaryItem.getVersion())
                        || (isNotBlank(param.getCurVer()) && isGrateOrEqualThan(param.getCurVer(), primaryItem.getVersion()));
            }).collect(Collectors.toList());

            indexPlusDTO.setPrimary(resultPrimaryList);
        }

        //过滤二级入口的生效版本
        if (CollectionUtils.isNotEmpty(secondaryList)) {
            List<LinkDTO> resultSecondaryList = secondaryList.stream().filter(secondaryItem -> {
                //如果不做版本限定或者当前版本大于等于生效版本
                return isBlank(secondaryItem.getVersion()) || Objects.equals("0", secondaryItem.getVersion())
                        || (isNotBlank(param.getCurVer()) && isGrateOrEqualThan(param.getCurVer(), secondaryItem.getVersion()));
            }).collect(Collectors.toList());

            indexPlusDTO.setSecondary(resultSecondaryList);
        }
    }

    private IndexPlusDTO fillGuideInfo(IndexPlusDTO indexPlus, IndexPlusParam param) {
        // 设置用户信息
        UserRegisterInfoDTO userRegisterInfoDTO = userIntegrationService.userRegisterInfo(param.getUserId());
        int day = 0;
        if (Objects.nonNull(userRegisterInfoDTO)) {
            IndexPlusUserInfoDTO userInfoDTO = new IndexPlusUserInfoDTO();
            userInfoDTO.setImgUrl(userRegisterInfoDTO.getHeadImg());
            userInfoDTO.setNickName(userRegisterInfoDTO.getNickname());
            userInfoDTO.setUserId(userRegisterInfoDTO.getUserId());

            // 已登录app天数
            if (Objects.nonNull(userRegisterInfoDTO.getIntoAppDays())) {
                day = userRegisterInfoDTO.getIntoAppDays();
            }
            indexPlus.setUserInfo(userInfoDTO);
        }

        // 设置文案 替换占位符
        indexPlus.setGuideContent(String.format(plusEntryProperties.getGuideContent(), day));

        return indexPlus;
    }

    private LinkDTO convert(NewsPublishEntry entry) {
        LinkDTO link = new LinkDTO();
        link.setId(entry.getId());
        link.setTitle(entry.getTitle());
        link.setSubTitle(entry.getSubTitle());
        link.setIcon(entry.getIconUrl());
        link.setProtocolUrl(entry.getProtocol());
        link.setVersion(entry.getVersion());
        return link;
    }
}
