package com.bxm.localnews.news.thread;

import com.bxm.localnews.integration.MessageService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.news.constant.RedisConfig;
import com.bxm.localnews.news.facade.service.ForumPostInnerService;
import com.bxm.localnews.news.model.vo.ForumPostVo;
import com.bxm.localnews.news.vo.ForumReplyInfo;
import com.bxm.localnews.news.vo.UserBean;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.fasterxml.jackson.core.type.TypeReference;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.concurrent.CountDownLatch;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 动态评论回复提醒 处理类
 */
@Data
@Slf4j
public class PostReplyThread implements Runnable {

    private final List<Long> postIds;

    private final CountDownLatch countDownLatch;

    private final RedisListAdapter redisListAdapter;

    private final UserIntegrationService userIntegrationService;

    private final MessageService messageService;

    private final ForumPostInnerService forumPostInnerService;

    /**
     * 推送文案1
     */
    private final static String COPY_WRITE_1 = "等%d人在你评论的内容下发表了%d条新评论";

    /**
     * 推送文案2
     */
    private final static String COPY_WRITE_2 = "在你评论的内容下发表了%d条新评论";

    public PostReplyThread(List<Long> postIds, CountDownLatch countDownLatch) {
        this.postIds = postIds;
        this.countDownLatch = countDownLatch;
        this.redisListAdapter = SpringContextHolder.getBean(RedisListAdapter.class);
        this.userIntegrationService = SpringContextHolder.getBean(UserIntegrationService.class);
        this.messageService = SpringContextHolder.getBean(MessageService.class);
        this.forumPostInnerService = SpringContextHolder.getBean(ForumPostInnerService.class);
    }

    @Override
    public void run() {
        try {
            if (!CollectionUtils.isEmpty(postIds)) {
                log.info("postId:{}", postIds);
                postIds.forEach(e -> {
                    KeyGenerator replyInfoKey = RedisConfig.FORUM_REPLY_INFO.copy().appendKey(e);
                    try {
                        //获取当前贴所有在缓存的数量
                        List<ForumReplyInfo> replyInfo = redisListAdapter.leftIndex(replyInfoKey, -1, new TypeReference<ForumReplyInfo>() {
                        });
                        dealDataAndSendPush(replyInfo, e);
                    } finally {
                        redisListAdapter.remove(replyInfoKey);
                    }
                });
            }
        } finally {
            countDownLatch.countDown();
        }
    }

    /**
     * 处理数据并发送推送
     */
    private void dealDataAndSendPush(List<ForumReplyInfo> replyInfo, Long postId) {
        //获取要发送推送的用户
        Set<Long> userIds = replyInfo.stream().map(ForumReplyInfo::getUserId).collect(Collectors.toSet());
        //批量获取用户信息
        List<UserBean> userBeans = userIntegrationService.batchUserInfo(Lists.newArrayList(userIds));
        Map<Long, UserBean> userInfos = userBeans.stream().collect(Collectors.toMap(UserBean::getId, Function.identity()));
        ForumPostVo forumPostVo = forumPostInnerService.getBriefInfo(postId);
        for (Long userId : userIds) {
            //若发送的用户的状态不正常时不发送评论回复提醒
            if (userInfos.get(userId).getState() != 1) {
                continue;
            }
            //获取当前用户最大的评论时间
            Optional<ForumReplyInfo> max = replyInfo.stream()
                    .filter(e -> Objects.equal(userId, e.getUserId()))
                    .max(Comparator.comparing(ForumReplyInfo::getReplyTime));
            if (max.isPresent()) {
                Date maxTime = max.get().getReplyTime();
                //查找比当前用户最大的评论时间大,且不为当前用户评论
                List<Long> infos = replyInfo.stream()
                        .filter(e -> e.getReplyTime().compareTo(maxTime) > 0)
                        .sorted(Comparator.comparing(ForumReplyInfo::getReplyTime))
                        .map(ForumReplyInfo::getUserId)
                        .collect(Collectors.toList());
                if (!CollectionUtils.isEmpty(infos)) {
                    //评论数量
                    int replyNum = infos.size();
                    //评论人数
                    long userNum = infos.stream().distinct().count();
                    if (userNum >= 1) {
                        UserBean replyUser = userInfos.get(infos.get(0));
                        String copyWrite;
                        if (userNum > 1) {
                            copyWrite = String.format(COPY_WRITE_1, userNum, replyNum);
                        } else {
                            copyWrite = String.format(COPY_WRITE_2, replyNum);
                        }
                        messageService.pushPostReply(userInfos.get(userId), copyWrite, forumPostVo, replyUser);
                    }
                }
            }
        }
    }


}
