package com.bxm.egg.mq.common.constant;

import com.bxm.newidea.component.tools.NumberUtils;


/**
 * 推送消息
 */
public enum PushMessageEnum {

    /**
     * 新版本不根据枚举定义消息类型
     * 但是需要跳转的话，还是需要定义MessageBehaviorEnum 所以这里进行定义默认的JUMP类型
     * 根据这个类型，客户端会执行跳转协议这个操作
     * <p>
     * <p>
     * 这里的消息类型有两个定义
     * 一个是推送的消息类型
     * 一个是消息中心的消息类型
     * <p>
     * 早先时候，这两个是混在一块的，例如发送一个系统公告的消息，则用户中心会有一条类型为0 推送也会受到一条为0的类型
     * 但这样有一个很不好的地方在于，如果类型越来越多，客户端也需要做支持。
     * 所以在新版之后，定义了一个105的推送类型，同时在类型的同一个层级增加一个protocol字段，供客户端路由跳转。
     * 客户端再接受到推送时，先不关心类型，直接判断protocol字段是否有值，有就直接跳转，否则再判断类型。
     * <p>
     * 这样一来，这个枚举下所有的类型，都只是消息本身的类型，和推送跳转无关了，跳转只需要确定一个protocol即可
     */
    JUMP_TYPE(105, "新版本推送需要跳转的类型", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),


    /**
     * 系统公告
     */
    SYSTEM_MSG(0, "系统公告", MessageBehaviorEnum.UN_JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    ORDINARY_MSG(1, "提示消息", MessageBehaviorEnum.UN_JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    OPEN_URL(10, "打开链接", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    PARTICIPATE_PRIVILEGE_SUCCESS(11, "参加活动成功", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    PRIVILEGE_OPEN_PRIZE(12, "活动开奖", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    PRIVILEGE_WINNING(13, "活动中奖", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    TASK_COMPLETED(15, "任务完成（日常/新手）", MessageBehaviorEnum.UN_JUMP, Boolean.TRUE, MessageTypeEnum.SYSTEM),
    NEW_MERCHANT(16, "商家提交信息成功", MessageBehaviorEnum.UN_JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    MERCHANT_VERIFY(18, "商家审核失败", MessageBehaviorEnum.UN_JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    POST_REVIEW_PASS(54, "帖子审核通过", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    ESSENCE_POST(55, "精华帖通知", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),
    IM_CHAT_ROOM_TIMING_RED_PACKET_MESSAGE(100, "群聊定时红包提醒", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.SYSTEM),


    INVITE(81, "邀请通知", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.INVITE),


    ADD_FUNS(98, "新增粉丝", MessageBehaviorEnum.JUMP, Boolean.TRUE, MessageTypeEnum.ADD_FUNS),


    /**
     * 今日热文
     */
    POST(8, "帖子", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.NEWS),

    /**
     * 互动消息
     */
    INTERACTION_MSG(5, "互动消息", MessageBehaviorEnum.UN_JUMP, Boolean.FALSE, MessageTypeEnum.INTERACTION),

    /**
     * 点赞信息
     */
    POST_LIKE(53, "帖子点赞", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.LIKE),
    BELONG_POST_LIKE(58, "帖子所属人点赞", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.LIKE),

    /**
     * 评论回复
     */
    POST_REPLY(52, "帖子回复", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.COMMENT),
    BELONG_POST_REPLY(57, "帖子所属人回复", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.COMMENT),

    /**
     * @我的内容
     */
    AT(107, "@我的评论", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.AT_ME),
    AT_SUB(108, "@我的二级评论", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.AT_ME),

    /**
     * 余额变动
     */
    FAIL_WITHDRAW(60, "您的微信号未实名认证，提现失败", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.BALANCE),
    ADD_CASH(61, "新增赏金", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.BALANCE),
    SUCCESS_WITHDRAW(62, "您的提现已成功打款", MessageBehaviorEnum.JUMP, Boolean.FALSE, MessageTypeEnum.BALANCE),
    ;

    private int type;

    private String desc;

    /**
     * 是否需要跳转（会影响在消息列表中显示的消息是否显示查看详情）
     */
    private MessageBehaviorEnum behavior;

    /**
     * 是否静默推送（无通知栏）
     */
    private Boolean silence;

    private MessageTypeEnum groupType;

    PushMessageEnum(int type, String desc, MessageBehaviorEnum behavior, Boolean silence, MessageTypeEnum groupType) {
        this.type = type;
        this.desc = desc;
        this.behavior = behavior;
        this.silence = silence;
        this.groupType = groupType;
    }

    public MessageTypeEnum getGroupType() {
        return groupType;
    }

    public static String getMessageDescByType(int type) {
        for (PushMessageEnum messageEnum : PushMessageEnum.values()) {
            if (messageEnum.getType() == type) {
                return messageEnum.getDesc();
            }
        }
        return null;
    }

    public static PushMessageEnum getMessageByType(int type) {
        for (PushMessageEnum messageEnum : PushMessageEnum.values()) {
            if (messageEnum.getType() == type) {
                return messageEnum;
            }
        }
        return PushMessageEnum.SYSTEM_MSG;
    }

    public static String getMessageDescByType(String type) {
        return getMessageDescByType(NumberUtils.parseToInt(type));
    }

    public static MessageBehaviorEnum getMessageBehaviorByType(int type) {
        for (PushMessageEnum messageEnum : PushMessageEnum.values()) {
            if (messageEnum.getType() == type) {
                return messageEnum.getBehavior();
            }
        }
        return null;
    }

    public static Boolean getMessageSilenceByType(int type) {
        for (PushMessageEnum messageEnum : PushMessageEnum.values()) {
            if (messageEnum.getType() == type) {
                return messageEnum.getSilence();
            }
        }
        return null;
    }

    public int getType() {
        return this.type;
    }

    public void setType(int type) {
        this.type = type;
    }

    public String getDesc() {
        return this.desc;
    }

    public void setDesc(String desc) {
        this.desc = desc;
    }

    public MessageBehaviorEnum getBehavior() {
        return behavior;
    }

    public void setBehavior(MessageBehaviorEnum behavior) {
        this.behavior = behavior;
    }

    public Boolean getSilence() {
        return silence;
    }

    public void setSilence(Boolean silence) {
        this.silence = silence;
    }
}
