package com.bxm.localnews.im.activity.impl;

import cn.hutool.core.date.DateUtil;
import com.bxm.localnews.im.activity.RedPacketDetailService;
import com.bxm.localnews.im.activity.RedPacketReminderService;
import com.bxm.localnews.im.activity.RedpacketReceiveService;
import com.bxm.localnews.im.config.RedPacketProperties;
import com.bxm.localnews.im.domain.activity.RedpacketPlanDetailMapper;
import com.bxm.localnews.im.domain.activity.RedpacketPlanMapper;
import com.bxm.localnews.im.dto.activity.RedPackageDto;
import com.bxm.localnews.im.dto.activity.RedPackageStatusDto;
import com.bxm.localnews.im.entity.activity.RedpacketPlanDetailEntity;
import com.bxm.localnews.im.entity.activity.RedpacketPlanEntity;
import com.bxm.localnews.im.entity.group.ImGroupEntity;
import com.bxm.localnews.im.enums.GroupStatusEnum;
import com.bxm.localnews.im.enums.RedpacketStatusEnum;
import com.bxm.localnews.im.group.GroupService;
import com.bxm.localnews.im.group.GroupTotalService;
import com.bxm.localnews.im.param.activity.RedPackageStatusParam;
import com.bxm.localnews.im.param.group.BaseGroupParam;
import com.bxm.localnews.im.thirdpart.IMSDKAdapter;
import com.bxm.localnews.im.thirdpart.message.RedPacketMessage;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Optional;

import static com.bxm.localnews.im.constant.RedPacketRedisConstant.REDPACKET_DETAIL_KEY;
import static com.bxm.newidea.component.tools.DateUtils.PATTERN_HOUR_MINUTE_FORMAT;

/**
 * @author liujia
 * @date 9/26/21 1:52 PM
 **/
@Service
@AllArgsConstructor
@Slf4j
public class RedPacketDetailServiceImpl implements RedPacketDetailService {

    private RedpacketPlanDetailMapper redpacketPlanDetailMapper;

    private RedpacketPlanMapper redpacketPlanMapper;

    private RedPacketReminderService redPacketReminderService;

    private RedpacketReceiveService redpacketReceiveService;

    private RedPacketProperties redPacketProperties;

    private RedisStringAdapter redisStringAdapter;

    private GroupService groupService;

    private GroupTotalService groupTotalService;

    private IMSDKAdapter imsdkAdapter;

    @Override
    public RedPackageDto getClosestRedpacket(BaseGroupParam param) {
        Date closestTime = DateUtils.addField(new Date(), Calendar.HOUR, 2);
        Date beforeTime = DateUtils.addField(new Date(), Calendar.HOUR, -1);

        List<RedpacketPlanDetailEntity> closestRedpacketList = redpacketPlanDetailMapper.getClosestRedpacket(param.getGroupId(),
                beforeTime,
                closestTime);

        if (closestRedpacketList.size() == 0) {
            return null;
        }

        RedpacketPlanDetailEntity closestRedpacket = null;

        //查找最近的可领取或待领取的红包
        Optional<RedpacketPlanDetailEntity> firstRedpacket = closestRedpacketList.stream().filter(redpacket -> {
            return RedpacketStatusEnum.WAIT.match(redpacket.getStatus())
                    || RedpacketStatusEnum.RECEIVE.match(redpacket.getStatus());
        }).findFirst();

        if (firstRedpacket.isPresent()) {
            closestRedpacket = firstRedpacket.get();
        } else {
            // 逆序查找最近的一个已领取的红包
            Optional<RedpacketPlanDetailEntity> firstFinishRedpacket = Lists.reverse(closestRedpacketList).stream().filter(redpacket -> {
                return RedpacketStatusEnum.FINISH.match(redpacket.getStatus());
            }).findFirst();

            if (firstFinishRedpacket.isPresent()) {
                closestRedpacket = firstFinishRedpacket.get();
            }
        }

        if (null == closestRedpacket) {
            return null;
        }

        String dayStr = "今天";
        if (!DateUtil.isSameDay(new Date(), closestRedpacket.getReceiveTime())) {
            if (DateUtils.before(closestRedpacket.getReceiveTime(), new Date())) {
                dayStr = "昨天";
            } else {
                dayStr = "明天";
            }
        }

        RedPackageDto packageInfo = RedPackageDto.builder()
                .title(redPacketProperties.getDefaultRedpacketTitle())
                .redpacketId(closestRedpacket.getId())
                .startTime(closestRedpacket.getReceiveTime())
                .status(closestRedpacket.getStatus())
                .dayStr(dayStr)
                .enableNotify(redPacketReminderService.enableRemind(param.getUserId(), param.getGroupId()))
                .build();

        if (RedpacketStatusEnum.RECEIVE.match(closestRedpacket.getStatus())) {
            packageInfo.setContent("粮食礼包派发中");
        } else if (RedpacketStatusEnum.WAIT.match(closestRedpacket.getStatus())) {
            String receiveTime = PATTERN_HOUR_MINUTE_FORMAT.get().format(closestRedpacket.getReceiveTime());
            packageInfo.setReceiveTimeStr(receiveTime);
            packageInfo.setContent("准时开抢");
        } else {
            String receiveTime = PATTERN_HOUR_MINUTE_FORMAT.get().format(closestRedpacket.getReceiveTime());
            packageInfo.setReceiveTimeStr(receiveTime);
            packageInfo.setContent("礼包派完了");
        }

        return packageInfo;
    }

    @Override
    public RedPackageStatusDto getUserRedpacketStatus(RedPackageStatusParam param) {
        RedPackageStatusDto status = getStatus(param.getRedPackageId());

        if (RedpacketStatusEnum.RECEIVE.match(status.getStatus())) {
            if (redpacketReceiveService.isReceived(param.getRedPackageId(), param.getUserId())) {
                // 客户端根据此值判断，直接进入详情
                status.setStatus(RedpacketStatusEnum.CLOSE.getCode());
            }
        }

        return status;
    }

    @Override
    public RedPackageStatusDto getStatus(Long detailId) {
        return loadRedpacketCache(detailId);
    }

    private RedPackageStatusDto loadRedpacketCache(Long detailId) {
        KeyGenerator detailKey = buildDetailKey(detailId);

        RedPackageStatusDto statusDto = redisStringAdapter.get(detailKey, RedPackageStatusDto.class);
        if (null == statusDto) {
            RedpacketPlanDetailEntity detailEntity = redpacketPlanDetailMapper.selectById(detailId);

            if (null == detailEntity) {
                statusDto = RedPackageStatusDto.builder()
                        .id(detailId)
                        .senderImg(redPacketProperties.getDefaultRedpacketHeadImg())
                        .title(redPacketProperties.getDefaultRedpacketTitle())
                        .content(RedpacketStatusEnum.CLOSE.getContent())
                        .status(RedpacketStatusEnum.FINISH.getCode())
                        .build();
            } else {
                statusDto = RedPackageStatusDto.builder()
                        .senderImg(redPacketProperties.getDefaultRedpacketHeadImg())
                        .title(redPacketProperties.getDefaultRedpacketTitle())
                        .id(detailId)
                        .relationId(detailEntity.getRelationId())
                        .status(detailEntity.getStatus())
                        .receiveTime(detailEntity.getReceiveTime())
                        .build();

                if (RedpacketStatusEnum.WAIT.match(detailEntity.getStatus())) {
                    statusDto.setContent(RedpacketStatusEnum.WAIT.getContent());
                } else if (RedpacketStatusEnum.RECEIVE.match(detailEntity.getStatus())) {
                    statusDto.setContent(RedpacketStatusEnum.RECEIVE.getContent());
                } else {
                    statusDto.setStatus(RedpacketStatusEnum.FINISH.getCode());
                    statusDto.setContent(RedpacketStatusEnum.FINISH.getContent());
                }
            }

            redisStringAdapter.set(detailKey, statusDto, 2 * 3600);
        }

        return statusDto;

    }

    private KeyGenerator buildDetailKey(Long detailId) {
        return REDPACKET_DETAIL_KEY.copy().appendKey(detailId);
    }

    private void removeRedpacketCache(Long detailId) {
        KeyGenerator detailKey = buildDetailKey(detailId);
        redisStringAdapter.remove(detailKey);
    }

    @Override
    public void removePlanDetail(Long planId) {
        log.info("移除红包投放计划[{}]对应的红包", planId);

        // 获取投放计划对应的处于激活状态的红包
        RedpacketPlanDetailEntity activeRedpacket = redpacketPlanDetailMapper.getActiveRedpacket(planId);
        if (activeRedpacket != null) {
            // 移除红包的提醒信息
            redPacketReminderService.removeRedpacketRemind(activeRedpacket.getId());
            // 移除红包的预领取队列
            redpacketReceiveService.removeReceiveQueue(activeRedpacket.getId());

            RedpacketPlanDetailEntity updateEntity = new RedpacketPlanDetailEntity();
            updateEntity.setId(activeRedpacket.getId());
            updateEntity.setStatus(RedpacketStatusEnum.CLOSE.getCode());

            redpacketPlanDetailMapper.updateById(updateEntity);

            // 清理缓存
            removeRedpacketCache(activeRedpacket.getId());
        }
    }

    @Override
    @Async
    public void closePlanDetail(Long redpacketDetailId) {
        RedpacketPlanDetailEntity detailEntity = redpacketPlanDetailMapper.selectById(redpacketDetailId);
        if (detailEntity != null) {
            RedpacketPlanDetailEntity updateEntity = new RedpacketPlanDetailEntity();
            updateEntity.setId(detailEntity.getId());
            updateEntity.setStatus(RedpacketStatusEnum.FINISH.getCode());
            updateEntity.setFinishTime(new Date());

            redpacketPlanDetailMapper.updateById(updateEntity);

            redPacketReminderService.removeRedpacketRemind(redpacketDetailId);
            redpacketReceiveService.removeReceiveQueue(redpacketDetailId);
            removeRedpacketCache(redpacketDetailId);

            // 群组内增加发放红包的统计数据
            groupTotalService.addRedpacket(detailEntity.getRelationId(), detailEntity.getTotalGrain());
        }
    }

    @Override
    public void processStatusPlan() {
        List<RedpacketPlanDetailEntity> waitRedpacketList = redpacketPlanDetailMapper.getWaitRedpacket(new Date());

        for (RedpacketPlanDetailEntity detailEntity : waitRedpacketList) {
            log.info("红包[{}]更新为可领取状态，预计时间：{},实际时间：{}",
                    detailEntity.getId(),
                    DateUtils.formatDateTime(detailEntity.getReceiveTime()),
                    DateUtils.formatDateTime(new Date()));

            RedpacketPlanDetailEntity updateEntity = new RedpacketPlanDetailEntity();
            updateEntity.setId(detailEntity.getId());
            updateEntity.setStatus(RedpacketStatusEnum.RECEIVE.getCode());

            redpacketPlanDetailMapper.updateById(updateEntity);

            removeRedpacketCache(detailEntity.getId());

            // 发送消息到群组
            RedPacketMessage redPacketMessage = RedPacketMessage.builder()
                    .name("定时粮食礼包")
                    .redPacketId(String.valueOf(detailEntity.getId()))
                    .typeName("定时红包")
                    .build();
            imsdkAdapter.sendGroupMessage(detailEntity.getRelationId(),
                    redPacketProperties.getChatRoomTimingRedPacketAssistantUserId(),
                    redPacketMessage);
        }
    }

    @Override
    public void createPlanDetail(Long planId) {
        RedpacketPlanEntity planEntity = redpacketPlanMapper.selectById(planId);

        ImGroupEntity groupEntity = groupService.getGroupEntity(planEntity.getRelationId());
        if (groupEntity == null || GroupStatusEnum.DISABLE.match(groupEntity.getStatus())) {
            log.info("群组[{}]已被禁用或不存在，无法开启新的红包", planEntity.getRelationId());
            return;
        }

        if (planEntity.getMaxTimes() > 0 && planEntity.getMaxTimes() <= planEntity.getCurrentTimes()) {
            RedpacketPlanEntity updatePlanEntity = new RedpacketPlanEntity();
            updatePlanEntity.setEnable(0);
            updatePlanEntity.setId(planEntity.getId());

            redpacketPlanMapper.updateById(updatePlanEntity);

            log.info("红包计划[{}]已经达到投放的最大次数", planId);
            return;
        }

        RedpacketPlanEntity updatePlanEntity = new RedpacketPlanEntity();
        updatePlanEntity.setId(planId);
        updatePlanEntity.setVersion(planEntity.getVersion());
        updatePlanEntity.setCurrentTimes(planEntity.getCurrentTimes());

        if (redpacketPlanMapper.addTimes(updatePlanEntity) > 0) {
            RedpacketPlanDetailEntity detailEntity = of(planEntity);

            redpacketPlanDetailMapper.insert(detailEntity);
            // 创建红包的预领取队列
            redpacketReceiveService.createReceiveQueue(detailEntity);
            // 创建红包对应的提醒信息
            redPacketReminderService.addRedpacketRemind(detailEntity);

            log.info("创建红包成功，投放计划：{}", planId);
        } else {
            log.error("创建红包详情失败，红包计划：{}", planId);
        }
    }

    private RedpacketPlanDetailEntity of(RedpacketPlanEntity planEntity) {
        Date clearNow = DateUtils.clearTimePart(new Date());
        clearNow = DateUtils.setField(clearNow, Calendar.HOUR, planEntity.getHour());
        clearNow = DateUtils.setField(clearNow, Calendar.MINUTE, planEntity.getMinute());

        RedpacketPlanDetailEntity entity = new RedpacketPlanDetailEntity();
        entity.setId(SequenceHolder.nextLongId());
        entity.setStatus(RedpacketStatusEnum.WAIT.getCode());
        entity.setReceiveTime(clearNow);
        entity.setCreateTime(new Date());

        entity.setTotalGrain(planEntity.getTotalGrain());
        entity.setMinNum(planEntity.getMinNum());
        entity.setMaxNum(planEntity.getMaxNum());
        entity.setPlanId(planEntity.getId());
        entity.setTimes(planEntity.getCurrentTimes() + 1);
        entity.setType(planEntity.getType());
        entity.setRelationId(planEntity.getRelationId());

        // 已到达时间，设置为可领取状态
        if (DateUtils.before(clearNow, new Date())) {
            entity.setStatus(RedpacketStatusEnum.RECEIVE.getCode());
        }

        return entity;
    }
}


























