package com.bxm.localnews.im.activity.impl;

import com.bxm.egg.message.integration.UserAccountIntegrationService;
import com.bxm.egg.message.integration.UserIntegrationService;
import com.bxm.egg.message.vo.UserInfoBean;
import com.bxm.localnews.im.activity.RedPacketDetailService;
import com.bxm.localnews.im.activity.RedpacketReceiveService;
import com.bxm.localnews.im.activity.strategy.RedpacketQueueContext;
import com.bxm.localnews.im.config.RedPacketProperties;
import com.bxm.localnews.im.constant.LogicGroupConstant;
import com.bxm.localnews.im.domain.activity.RedpacketReceiveRecordMapper;
import com.bxm.localnews.im.dto.activity.RedPackageDetailDto;
import com.bxm.localnews.im.dto.activity.RedPackageStatusDto;
import com.bxm.localnews.im.dto.activity.RedPacketReceiveDTO;
import com.bxm.localnews.im.entity.activity.RedpacketPlanDetailEntity;
import com.bxm.localnews.im.entity.activity.RedpacketReceiveRecordEntity;
import com.bxm.localnews.im.enums.RedpacketQueueStrategyEnum;
import com.bxm.localnews.im.enums.RedpacketStatusEnum;
import com.bxm.localnews.im.param.activity.OpenRedPackageParam;
import com.bxm.localnews.im.param.activity.RedPackageDetailParam;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.redis.*;
import com.bxm.newidea.component.strategy.ReturnedStrategyExecutor;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.bxm.localnews.im.constant.RedPacketRedisConstant.*;

/**
 * @author liujia
 * @date 9/26/21 4:12 PM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class RedpacketReceiveServiceImpl implements RedpacketReceiveService {

    private RedpacketReceiveRecordMapper redpacketReceiveRecordMapper;

    private RedPacketProperties redPacketProperties;

    private RedisSetAdapter redisSetAdapter;

    private RedisListAdapter redisListAdapter;

    private RedisStringAdapter redisStringAdapter;

    private DistributedLock distributedLock;

    private ReturnedStrategyExecutor returnedStrategyExecutor;

    private UserAccountIntegrationService userAccountIntegrationService;

    private UserIntegrationService userIntegrationService;

    @Override
    public RedPackageDetailDto executeOpen(OpenRedPackageParam param) {
        RedPackageDetailDto detailDto = RedPackageDetailDto.builder().build();

        KeyGenerator lockKey = REDPACKET_OPEN_LOCK.copy().appendKey(param.getRedPackageId()).appendKey(param.getUserId());
        if (distributedLock.lock(lockKey.gen())) {
            RedPackageStatusDto status = getRedPacketService().getStatus(param.getRedPackageId());
            if (!RedpacketStatusEnum.RECEIVE.match(status.getStatus())) {
                log.warn("红包[{}]处于不可领取的状态，触发了开启红包", param.getRedPackageId());
                return detailDto;
            }


            RedPackageDetailParam detailParam = new RedPackageDetailParam();
            detailParam.setRedPackageId(param.getRedPackageId());
            detailParam.setUserId(param.getUserId());

            // 判断当前用户是否领取过了
            if (isReceived(param.getRedPackageId(), param.getUserId())) {
                detailDto = getDetail(detailParam);
                detailDto.setPromptText("你已经领取过该红包，不能重复领取哦");
            } else {
                // 尝试从预领取队列中获得一个红包，如果获取成功，则认为领取成功
                KeyGenerator queueKey = buildQueueKey(param.getRedPackageId());
                Integer randomItem = redisListAdapter.leftPop(queueKey, Integer.class);

                if (randomItem == null) {
                    detailDto = getDetail(detailParam);
                    detailDto.setPromptText("来晚了，粮食已经被领完了");
                } else {
                    addRecord(param, randomItem);

                    detailDto = getDetail(detailParam);
                    detailDto.setPromptText("粮食已存入您的账户");

                    afterReceiveOver(param.getRedPackageId());
                }
            }


            distributedLock.unlock(lockKey.gen());
        }

        return detailDto;
    }

    @Override
    public boolean isReceived(Long detailId, Long userId) {
        KeyGenerator receiveKey = buildReceiveUserKey(detailId);
        return redisSetAdapter.exists(receiveKey, userId);
    }

    /**
     * 领取以后的后续处理
     * 扣除可领取数量，如果扣到了0，说明领取完毕，则关闭红包
     */
    private void afterReceiveOver(Long detailId) {
        KeyGenerator queueNumKey = buildQueueNumKey(detailId);
        Long lastNum = redisStringAdapter.decrement(queueNumKey);

        if (lastNum <= 0) {
            getRedPacketService().closePlanDetail(detailId);
        }
    }

    private RedPacketDetailService getRedPacketService() {
        return SpringContextHolder.getBean(RedPacketDetailService.class);
    }

    private void addRecord(OpenRedPackageParam param, Integer receiveNum) {
        RedpacketReceiveRecordEntity entity = new RedpacketReceiveRecordEntity();
        entity.setId(SequenceHolder.nextLongId());
        entity.setCreateTime(new Date());
        entity.setDetailId(param.getRedPackageId());
        entity.setNum(receiveNum);
        entity.setUserId(param.getUserId());

        redpacketReceiveRecordMapper.insert(entity);

        //调用账户服务，给用户增加粮食
        Message message = userAccountIntegrationService.addRedpacketGrain(param.getUserId(), receiveNum);
        if (!message.isSuccess()) {
            log.error("用户[{}]领取群红包后，发放粮食[{}]失败", param.getUserId(), receiveNum);
        }

        // 增加领取记录，防止重复领取
        KeyGenerator receiveKey = buildReceiveUserKey(param.getRedPackageId());
        redisSetAdapter.add(receiveKey, param.getUserId());
    }

    private KeyGenerator buildReceiveUserKey(Long detailId) {
        return REDPACKET_RECEIVE_CACHE_KEY.copy().appendKey(detailId);
    }

    private KeyGenerator buildQueueKey(Long detailId) {
        return REDPACKET_RECEIVE_QUEUE_KEY.copy().appendKey(detailId);
    }

    private KeyGenerator buildQueueNumKey(Long detailId) {
        return REDPACKET_RECEIVE_QUEUE_NUM_KEY.copy().appendKey(detailId);
    }

    @Override
    public RedPackageDetailDto getDetail(RedPackageDetailParam param) {
        List<RedpacketReceiveRecordEntity> allReceiveList = redpacketReceiveRecordMapper.getAll(param.getRedPackageId());

        Optional<RedpacketReceiveRecordEntity> currentUser = allReceiveList.stream().filter(receive -> {
            return Objects.equals(receive.getUserId(), param.getUserId());
        }).findFirst();

        Integer currentUserReceiveNum = null;
        if (currentUser.isPresent()) {
            currentUserReceiveNum = currentUser.get().getNum();
        }

        List<RedPacketReceiveDTO> packetReceives = allReceiveList.stream().filter(receive -> {
            return !Objects.equals(receive.getUserId(), param.getUserId());
        }).map(this::of).collect(Collectors.toList());

        // 当前用户排到首位
        currentUser.ifPresent(recordEntity -> packetReceives.add(0, this.of(recordEntity)));

        return RedPackageDetailDto.builder()
                .title(redPacketProperties.getDefaultRedpacketTitle())
                .senderImg(redPacketProperties.getDefaultRedpacketHeadImg())
                .receiveNum(currentUserReceiveNum)
                .otherReceiveList(packetReceives)
                .build();
    }

    private RedPacketReceiveDTO of(RedpacketReceiveRecordEntity recordEntity) {
        UserInfoBean userInfo = userIntegrationService.getUserInfo(recordEntity.getUserId());

        RedPacketReceiveDTO receiveDTO = new RedPacketReceiveDTO();
        receiveDTO.setNum(recordEntity.getNum());
        if (recordEntity.getCreateTime() != null) {
            receiveDTO.setReceiveTime(DateUtils.PATTERN_HOUR_MINUTE_SECEND_FORMAT.get().format(recordEntity.getCreateTime()));
        }
        receiveDTO.setUserId(recordEntity.getUserId());
        receiveDTO.setHeadImg(userInfo.getHeadImg());
        receiveDTO.setNickName(userInfo.getNickname());

        return receiveDTO;
    }

    @Override
    public void createReceiveQueue(RedpacketPlanDetailEntity detailEntity) {
        RedpacketQueueContext context = RedpacketQueueContext.builder()
                .strategy(RedpacketQueueStrategyEnum.RANDOM)
                .detailEntity(detailEntity)
                .build();

        KeyGenerator queueKey = buildQueueKey(detailEntity.getId());

        List<Integer> queueList = returnedStrategyExecutor.execute(LogicGroupConstant.RAD_PACKET_QUEUE, context);
        Integer[] queueArray = queueList.toArray(new Integer[]{});
        redisListAdapter.rightPush(queueKey, queueArray);

        KeyGenerator queueNumKey = buildQueueNumKey(detailEntity.getId());
        redisStringAdapter.set(queueNumKey, queueList.size());
    }

    @Override
    public void removeReceiveQueue(Long detailId) {
        KeyGenerator queueKey = buildQueueKey(detailId);
        redisListAdapter.remove(queueKey);

        KeyGenerator queueNumKey = buildQueueNumKey(detailId);
        redisStringAdapter.remove(queueNumKey);

        // 清除领取记录缓存
        KeyGenerator receiveKey = buildReceiveUserKey(detailId);
        redisSetAdapter.remove(receiveKey);
    }

}
