package com.bxm.localnews.im.chat.impl;

import com.bxm.component.mybatis.utils.MybatisBatchBuilder;
import com.bxm.localnews.im.chat.ConsumeService;
import com.bxm.localnews.im.domain.ChatMessageMapper;
import com.bxm.localnews.im.enums.ChannelTypeEnum;
import com.bxm.localnews.im.param.RongCloudParam;
import com.bxm.localnews.im.thirdpart.MsgContentProcesser;
import com.bxm.localnews.im.thirdpart.RCProcessorFactory;
import com.bxm.localnews.im.thirdpart.enums.MsgTypeEnum;
import com.bxm.localnews.im.vo.IMMessageBean;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.thread.NamedThreadFactory;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static com.bxm.localnews.im.constant.ImRedisKey.MSG_SET;

@Service
@Slf4j
public class ConsumeServiceImpl implements ConsumeService, ApplicationRunner {

    private final RedisSetAdapter redisSetAdapter;

    private final static long FETCH_NUM = 500L;

    @Autowired
    public ConsumeServiceImpl(RedisSetAdapter redisSetAdapter) {
        this.redisSetAdapter = redisSetAdapter;
    }

    @Override
    public void write(RongCloudParam param) {
        if (log.isDebugEnabled()) {
            log.debug("新增一条通讯消息：[{}]", param);
        }

        redisSetAdapter.add(MSG_SET, param);
    }

    /**
     * 读取消息队列，批量获取后进行数据写入与消息处理
     */
    private void consume() {
        List<RongCloudParam> msgList = redisSetAdapter.pop(MSG_SET, FETCH_NUM, RongCloudParam.class);

        if (msgList.size() > 0) {
            List<IMMessageBean> messages = convert(msgList);

            //批量插入私聊消息
            batchInsertPersonMessage(messages);
            //批量插入群组消息
            batchInsertGroupMessage(messages);

            if (log.isDebugEnabled()) {
                log.debug("消息处理完成，消息数量：[{}]", msgList.size());
            }

            if (msgList.size() >= FETCH_NUM) {
                consume();
            }
        }
    }

    /**
     * 批量插入群组消息
     *
     * @param messages 接受到的消息
     */
    private void batchInsertGroupMessage(List<IMMessageBean> messages) {
        //获取当前用户消息是私聊消息
        List<IMMessageBean> chatRoomList = messages.stream()
                .filter(e -> StringUtils.equals(ChannelTypeEnum.GROUP.getTypeName(), e.getChannelType()))
                .collect(Collectors.toList());
        //批量写入到数据库
        MybatisBatchBuilder.create(ChatMessageMapper.class, chatRoomList).run(ChatMessageMapper::insertGroupMessage);
    }

    /**
     * 批量插入私聊消息
     *
     * @param messages 接收到的消息
     */
    private void batchInsertPersonMessage(List<IMMessageBean> messages) {
        //获取当前用户消息是私聊消息
        List<IMMessageBean> personList = messages.stream()
                .filter(e -> StringUtils.equals(ChannelTypeEnum.PERSON.getTypeName(), e.getChannelType()))
                .collect(Collectors.toList());
        //批量写入到数据库
        MybatisBatchBuilder.create(ChatMessageMapper.class, personList).run(ChatMessageMapper::insert);
    }

    private List<IMMessageBean> convert(List<RongCloudParam> msgList) {
        List<IMMessageBean> messages = Lists.newArrayList();

        for (RongCloudParam paramMsg : msgList) {
            MsgContentProcesser processor = RCProcessorFactory.get(paramMsg.getObjectName());
            IMMessageBean message = build(paramMsg, processor);
            processor.callback(message);
            if (MsgTypeEnum.CONTENT.equals(processor.msgType())) {
                messages.add(message);
            }
        }

        return messages;
    }

    /**
     * 将融创传递的消息对象转换为通用的消息类型
     *
     * @param paramMsg  融创消息对象
     * @param processor 对应消息类型的处理器
     * @return 转换结果
     */
    private IMMessageBean build(RongCloudParam paramMsg, MsgContentProcesser processor) {
        String briefContent = processor.parseBrief(paramMsg.getContent());
        String parseContent = processor.parse(paramMsg.getContent());

        return IMMessageBean.builder()
                .msgId(paramMsg.getMsgUID())
                .fromUserId(Long.valueOf(paramMsg.getFromUserId()))
                .toUserId(Long.valueOf(paramMsg.getToUserId()))
                .content(paramMsg.getContent())
                .briefContent(briefContent)
                .parseContent(parseContent)
                .msgTimestamp(new Date(Long.valueOf(paramMsg.getMsgTimestamp())))
                .channelType(paramMsg.getChannelType())
                .groupUserIds(StringUtils.join(paramMsg.getGroupUserIds(), ","))
                .sensitiveType(paramMsg.getSensitiveType())
                .source(paramMsg.getSource())
                .msgType(paramMsg.getObjectName())
                .build();
    }

    @Override
    public void run(ApplicationArguments args) {
        ScheduledExecutorService threadPool = new ScheduledThreadPoolExecutor(1, new NamedThreadFactory("consume-message"));
        threadPool.scheduleAtFixedRate(this::consume, 3, 3, TimeUnit.SECONDS);
    }
}
