package com.bxm.localnews.im.chat.impl;

import com.bxm.egg.common.url.ProtocolFactory;
import com.bxm.egg.message.facade.service.MessageFacadeService;
import com.bxm.egg.message.integration.UserAccountIntegrationService;
import com.bxm.egg.message.integration.UserIntegrationService;
import com.bxm.egg.message.vo.UserInfoBean;
import com.bxm.egg.mq.common.model.dto.PushMessage;
import com.bxm.egg.mq.common.param.DingtalkMessage;
import com.bxm.localnews.im.chat.IMService;
import com.bxm.localnews.im.config.IMProperties;
import com.bxm.localnews.im.domain.ChatMessageMapper;
import com.bxm.localnews.im.integration.ConfigClientIntegrationService;
import com.bxm.localnews.im.param.chat.GiftParam;
import com.bxm.localnews.im.thirdpart.IMSDKAdapter;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import static com.bxm.localnews.im.constant.ImRedisKey.GIFT_RECORD_KEY;
import static com.bxm.localnews.im.constant.ImRedisKey.TOKEN;

@Service
@Slf4j
@AllArgsConstructor
public class IMServiceImpl implements IMService {

    private final IMSDKAdapter imsdkAdapter;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final RedisSetAdapter redisSetAdapter;

    private final UserIntegrationService userIntegrationService;

    private final UserAccountIntegrationService userAccountIntegrationService;

    private final ChatMessageMapper chatMessageMapper;

    private final ConfigClientIntegrationService clientIntegrationService;

    private final MessageFacadeService messageFacadeService;

    private final IMProperties imProperties;

    private KeyGenerator buildKey(Long userId) {
        return TOKEN.copy().appendKey(userId % 10);
    }

    @Override
    public String getToken(Long userId) {
        log.debug("获取token开始，用户:[{}]", userId);
        Preconditions.checkArgument(null != userId);

        KeyGenerator key = buildKey(userId);

        String token = redisHashMapAdapter.get(key, userId.toString(), String.class);
        if (null == token) {
            UserInfoBean user = userIntegrationService.getUserInfo(userId);

            //用户信息无效的数据直接返回
            if (null == user
                    || StringUtils.isEmpty(user.getNickname())
                    || StringUtils.isEmpty(user.getHeadImg())) {
                return StringUtils.EMPTY;
            }

            token = imsdkAdapter.token(user);
            if (StringUtils.isNotBlank(token)) {
                redisHashMapAdapter.put(key, userId.toString(), token);
            }
        }
        log.debug("获取用户:[{}],token:[{}]", userId, token);
        return token;
    }

    @Override
    public void updateUser(Long userId) {
        imsdkAdapter.update(userIntegrationService.getUserInfo(userId));
    }

    @Override
    public Message executeGift(GiftParam param) {
        KeyGenerator giftKey = GIFT_RECORD_KEY.copy()
                .appendKey(param.getUserId())
                .appendKey(DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));

        if (redisSetAdapter.exists(giftKey, param.getTargetUserId())) {
            return Message.build(false, "今天已给好友赠送过了~明天再送吧");
        }

        // 尝试扣除用户的粮食
        Message message = userAccountIntegrationService.reduce(param.getUserId(), imProperties.getReduceGiftGrain());

        if (message.isSuccess()) {
            message = userAccountIntegrationService.addGiftGrain(param.getTargetUserId(), imProperties.getReduceGiftGrain());
            if (message.isSuccess()) {
                // 增加赠送的记录
                redisSetAdapter.add(giftKey, param.getTargetUserId());
                redisSetAdapter.expireWithDisrupt(giftKey, DateUtils.getCurSeconds());

                // 发送推送
                UserInfoBean userInfo = userIntegrationService.getUserInfo(param.getUserId());

                String pushContent = userInfo.getNickname() + "给你送来了" + imProperties.getReduceGiftGrain() + "g粮食，赶快去答谢吧~";
                PushMessage pushMessage = PushMessage.build("好友给你赠粮了", pushContent)
                        .assign(param.getTargetUserId());
                String toFarmUrl = clientIntegrationService.getConfigClientFacadeService().getSpecificConfig("toFarmUrl");
                String protocol = ProtocolFactory.appH5().url(toFarmUrl).build();
                pushMessage.getPayloadInfo().setProtocol(protocol);

                messageFacadeService.sendPushMessage(pushMessage);

                return Message.build(true, "赠送成功，友谊更进一步");
            }

            log.error("给好友增加粮食失败,请求参数：{}", JSON.toJSONString(param));
        } else {
            return Message.build(false, "您的粮食不足，去做任务领粮食吧~");
        }

        return Message.build();
    }

    @Override
    public void countAndPush() {
        int monitorNum = imProperties.getMonitorNum();

        Date startTime = DateUtils.clearTimePart(new Date());
        Date endTime = DateUtils.setField(startTime, Calendar.HOUR_OF_DAY, imProperties.getTriggerHour());

        int total = chatMessageMapper.sendUserCount(startTime, endTime);
        List<Long> monitorUserIds = chatMessageMapper.queryMonitorUserIds(startTime, endTime, monitorNum);

        StringBuilder content = new StringBuilder();
        content.append(DateUtils.formatDateTime(endTime)).append("私聊情况汇总=======\n");
        content.append("0点至").append(imProperties.getTriggerHour()).append("参与私聊人数：[")
                .append(total).append("]\n");
        content.append("0点至").append(imProperties.getTriggerHour()).append("参与私聊数量超过")
                .append(monitorNum).append("次以上的用户：\n");

        if (monitorUserIds.size() > 0) {
            for (Long userId : monitorUserIds) {
                content.append("[").append(userId).append("]\n");
            }
        } else {
            content.append("[无]\n");
        }

        messageFacadeService.sendDingtalk(DingtalkMessage.builder()
                .content(content.toString())
                .scene("IM")
                .build());
    }
}
