package com.bxm.localnews.im.group.impl;

import com.bxm.egg.common.param.BaseAreaCodeParam;
import com.bxm.egg.user.facade.service.UserEggInviteFacadeService;
import com.bxm.localnews.im.bo.GroupInfoBO;
import com.bxm.localnews.im.bo.ImGroupMemberInfoBO;
import com.bxm.localnews.im.config.IMProperties;
import com.bxm.localnews.im.convert.ImGroupConverter;
import com.bxm.localnews.im.domain.group.ImGroupMapper;
import com.bxm.localnews.im.domain.group.ImGroupMemberMapper;
import com.bxm.localnews.im.domain.group.ImGroupTotalMapper;
import com.bxm.localnews.im.dto.group.GroupNoticeDTO;
import com.bxm.localnews.im.dto.group.GroupRuntimeInfoDTO;
import com.bxm.localnews.im.dto.group.ImCrumbDTO;
import com.bxm.localnews.im.dto.group.ImGroupInfoDTO;
import com.bxm.localnews.im.entity.group.ImGroupMemberEntity;
import com.bxm.localnews.im.entity.group.ImGroupTotalEntity;
import com.bxm.localnews.im.enums.GroupStatusEnum;
import com.bxm.localnews.im.group.GroupCompositeService;
import com.bxm.localnews.im.group.GroupNoticeService;
import com.bxm.localnews.im.group.GroupService;
import com.bxm.localnews.im.param.group.BaseGroupParam;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Lists;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import java.util.List;
import java.util.Objects;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @author liujia
 * @date 9/17/21 4:18 PM
 **/
@Service
@Slf4j
public class GroupCompositeServiceImpl implements GroupCompositeService {

    private GroupNoticeService groupNoticeService;

    private ImGroupMemberMapper imGroupMemberMapper;

    private ImGroupMapper imGroupMapper;

    private IMProperties imProperties;

    private ImGroupTotalMapper imGroupTotalMapper;

    private GroupService groupService;

    private LoadingCache<String, List<String>> headImgCache;

    private UserEggInviteFacadeService userEggInviteFacadeService;

    public GroupCompositeServiceImpl(GroupNoticeService groupNoticeService,
                                     ImGroupMemberMapper imGroupMemberMapper,
                                     ImGroupMapper imGroupMapper,
                                     IMProperties imProperties,
                                     ImGroupTotalMapper imGroupTotalMapper,
                                     GroupService groupService,
                                     UserEggInviteFacadeService userEggInviteFacadeService) {
        this.groupNoticeService = groupNoticeService;
        this.imGroupMemberMapper = imGroupMemberMapper;
        this.imGroupMapper = imGroupMapper;
        this.imProperties = imProperties;
        this.imGroupTotalMapper = imGroupTotalMapper;
        this.groupService = groupService;
        this.userEggInviteFacadeService = userEggInviteFacadeService;
    }

    @PostConstruct
    private void init() {
        headImgCache = CacheBuilder.newBuilder()
                .expireAfterWrite(1, TimeUnit.HOURS)
                .maximumSize(200)
                .build(new CacheLoader<String, List<String>>() {
                    @Override
                    public List<String> load(String key) throws Exception {
                        List<ImGroupMemberEntity> joinMemberList = imGroupMemberMapper.queryMember(null,
                                null,
                                null,
                                5);

                        if (joinMemberList.size() > 0) {
                            return joinMemberList.stream()
                                    .map(ImGroupMemberEntity::getHeadImg)
                                    .collect(Collectors.toList());
                        }
                        return Lists.newArrayList();
                    }
                });
    }

    @Override
    public ImGroupInfoDTO get(BaseGroupParam param) {
        ImGroupMemberEntity memberInfo = imGroupMemberMapper.getMemberInfo(param.getGroupId(), param.getUserId());
        if (memberInfo == null) {
            memberInfo = new ImGroupMemberEntity();
        }

        GroupNoticeDTO enableNotice = groupNoticeService.getEnableNotice(param.getGroupId());

        List<ImGroupMemberEntity> joinMemberList = imGroupMemberMapper.queryActiveMember(param.getGroupId(), 19);

        ImGroupTotalEntity totalEntity = imGroupTotalMapper.selectById(param.getGroupId());

        return ImGroupInfoDTO.builder()
                .notice(enableNotice)
                .mute(Objects.equals(memberInfo.getMuted(), 1))
                .members(joinMemberList.stream().map(ImGroupConverter.INSTANCE::of).collect(Collectors.toList()))
                .full(totalEntity.getCurrentMemberNum() >= totalEntity.getMaxMemberNum())
                .build();
    }

    @Override
    public ImCrumbDTO getCrumb(BaseAreaCodeParam param) {
        // 查询用户已经加入的群聊
        ImGroupMemberInfoBO userActiveGroup = imGroupMemberMapper.getUserActiveGroup(param.getUserId());
        boolean joinButtonVisible = null == userActiveGroup;

        String groupName = "本地没有群组";
        Long groupId = null;
        GroupRuntimeInfoDTO runtimeInfoDTO = null;

        if (null == userActiveGroup) {
            // 优先加入推荐人所在的群组
            GroupRuntimeInfoDTO inviteRuntime = queryInviteUserGroup(param.getUserId());

            if (null != inviteRuntime) {
                groupId = inviteRuntime.getGroupId();
                groupName = inviteRuntime.getTitle();
            } else {
                // 推荐一个群给用户加入
                List<GroupInfoBO> groupList = imGroupMapper.getEnableGroupByArea(param.getAreaCode(), param.getUserId());

                boolean hit = false;

                for (GroupInfoBO groupInfoBO : groupList) {
                    if (GroupStatusEnum.NORMAL.match(groupInfoBO.getStatus()) && groupInfoBO.getCurrentNum() < groupInfoBO.getMaxNum()) {
                        groupId = groupInfoBO.getGroupId();
                        groupName = groupInfoBO.getGroupName();
                        hit = true;
                        break;
                    }
                }

                // 就算没有可以加入的群组，找一个满员的群组作为群名称
                if (!hit) {
                    for (GroupInfoBO groupInfoBO : groupList) {
                        if (GroupStatusEnum.FULL.match(groupInfoBO.getStatus()) || groupInfoBO.getCurrentNum() >= groupInfoBO.getMaxNum()) {
                            groupId = groupInfoBO.getGroupId();
                            groupName = groupInfoBO.getGroupName();
                            break;
                        }
                    }
                    joinButtonVisible = false;
                }
            }

        } else {
            groupId = userActiveGroup.getGroupId();
            groupName = userActiveGroup.getGroupName();

            runtimeInfoDTO = groupService.loadGroupRuntimeInfo(groupId);
        }

        // 获取当前地区的可用群组信息
        return ImCrumbDTO.builder()
                .groupId(groupId)
                .groupTitle(groupName)
                .joinButtonVisible(joinButtonVisible)
                .runtimeInfo(runtimeInfoDTO)
                .backgroundImgUrl(imProperties.getCrumbImgUrl())
                .memberHeadImgList(headImgCache.getUnchecked(param.getAreaCode()))
                .build();
    }

    private GroupRuntimeInfoDTO queryInviteUserGroup(Long userId) {
        Long inviteUserId = userEggInviteFacadeService.getInviteUserId(userId);
        if (null != inviteUserId) {
            ImGroupMemberInfoBO inviteUserGroup = imGroupMemberMapper.getUserActiveGroup(inviteUserId);
            if (null != inviteUserGroup) {
                // 群组未满才可以加入
                GroupRuntimeInfoDTO inviteUserGroupRuntimeInfo = groupService.loadGroupRuntimeInfo(inviteUserGroup.getGroupId());
                if (inviteUserGroupRuntimeInfo.getCurrentNum() < inviteUserGroupRuntimeInfo.getTotalNum()) {
                    return inviteUserGroupRuntimeInfo;
                }
            }
        }
        return null;
    }
}
