package com.bxm.localnews.im.group.impl;

import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.component.mybatis.dto.PlusPageModelDTO;
import com.bxm.egg.message.dto.LocationDTO;
import com.bxm.egg.message.integration.LocationIntegrationService;
import com.bxm.egg.message.integration.UserIntegrationService;
import com.bxm.egg.message.vo.UserInfoBean;
import com.bxm.localnews.im.config.GroupProperties;
import com.bxm.localnews.im.config.RedPacketProperties;
import com.bxm.localnews.im.convert.ImGroupConverter;
import com.bxm.localnews.im.domain.group.ImGroupComplainMapper;
import com.bxm.localnews.im.domain.group.ImGroupMapper;
import com.bxm.localnews.im.domain.group.ImGroupMemberMapper;
import com.bxm.localnews.im.dto.group.GroupManageListItemDTO;
import com.bxm.localnews.im.dto.group.GroupRuntimeInfoDTO;
import com.bxm.localnews.im.entity.group.ImGroupComplainEntity;
import com.bxm.localnews.im.entity.group.ImGroupEntity;
import com.bxm.localnews.im.entity.group.ImGroupMemberEntity;
import com.bxm.localnews.im.enums.GroupActionEnum;
import com.bxm.localnews.im.enums.GroupMemberTypeEnum;
import com.bxm.localnews.im.enums.GroupStatusEnum;
import com.bxm.localnews.im.group.*;
import com.bxm.localnews.im.param.group.ChangeEnableParam;
import com.bxm.localnews.im.param.group.ComplainGroupParam;
import com.bxm.localnews.im.param.group.GroupManageQueryParam;
import com.bxm.localnews.im.param.group.SaveGroupParam;
import com.bxm.localnews.im.thirdpart.IMSDKAdapter;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.dto.IPageModel;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.stereotype.Service;

import java.util.Date;
import java.util.List;
import java.util.Objects;

import static com.bxm.localnews.im.constant.ImRedisKey.GROUP_RUNTIME_CACHE_KEY;

/**
 * @author liujia
 * @date 9/17/21 2:32 PM
 **/
@Service
@Slf4j
@AllArgsConstructor
public class GroupServiceImpl implements GroupService {

    private ImGroupMapper imGroupMapper;

    private GroupTotalService groupTotalService;

    private ImGroupComplainMapper imGroupComplainMapper;

    private IMSDKAdapter imsdkAdapter;

    private LocationIntegrationService locationIntegrationService;

    private ImGroupMemberMapper imGroupMemberMapper;

    private GroupProperties groupProperties;

    private GroupActionService groupActionService;

    private RedPacketProperties redPacketProperties;

    private RedisStringAdapter redisStringAdapter;

    private UserIntegrationService userIntegrationService;

    private GroupMemberHeadImgService groupMemberHeadImgService;

    private GroupMemberService getGroupMemberService() {
        return SpringContextHolder.getBean(GroupMemberService.class);
    }

    @Override
    public IPageModel<GroupManageListItemDTO> query(GroupManageQueryParam param) {
        Page<GroupManageListItemDTO> page = new Page<>(param.getPageNum(), param.getPageSize());
        IPage<GroupManageListItemDTO> list = imGroupMapper.queryByPage(page, param);

        for (GroupManageListItemDTO record : list.getRecords()) {
            LocationDTO location = locationIntegrationService.getLocationByGeocode(record.getAreaCode());
            record.setAreaName(location.getName());
        }
        return PlusPageModelDTO.build(list);
    }

    @Override
    public Message changeStatus(ChangeEnableParam param) {
        ImGroupEntity entity = new ImGroupEntity();
        entity.setEnable(param.getEnable() ? 1 : 0);
        entity.setId(param.getGroupId());

        return Message.build(imGroupMapper.updateById(entity));
    }

    @Override
    public Message changeGlobalMute(ChangeEnableParam param) {
        ImGroupEntity entity = new ImGroupEntity();
        entity.setGlobalMute(param.getEnable() ? 0 : 1);
        entity.setId(param.getGroupId());

        Message message = Message.build(imGroupMapper.updateById(entity));

        if (message.isSuccess()) {
            reloadRuntimeCache(param.getGroupId());
        }

        return message;
    }

    @Override
    public Message saveOrUpdate(SaveGroupParam param) {
        ImGroupEntity existsEntity = imGroupMapper.selectById(param.getGroupId());

        ImGroupEntity imGroupEntity = ImGroupConverter.INSTANCE.of(param);

        Message message = Message.build();

        if (existsEntity != null) {
            // 更新基础信息
            imGroupMapper.updateById(imGroupEntity);
            message = afterUpdate(existsEntity, imGroupEntity);
        } else {
            // 保存数据
            imGroupEntity.setStatus(GroupStatusEnum.NORMAL.getCode());
            imGroupEntity.setCreateTime(new Date());
            imGroupEntity.setImgUrl(groupProperties.getDefaultGroupImg());

            imGroupMapper.insert(imGroupEntity);

            message.append(afterCreate(imGroupEntity));
        }

        initGroupManageUser();

        return message;
    }

    private Message afterCreate(ImGroupEntity imGroupEntity) {
        // 初始化统计数据
        groupTotalService.init(imGroupEntity);
        // 调用SDK，创建群组
        Message message = imsdkAdapter.createGroup(imGroupEntity.getId(),
                imGroupEntity.getName(),
                groupProperties.getManageUserId(),
                redPacketProperties.getChatRoomTimingRedPacketAssistantUserId());

        if (message.isSuccess()) {
            // 将群管理、红包发放人员加入到群组
            getGroupMemberService().addMember(imGroupEntity.getId(),
                    imGroupEntity.getName(),
                    groupProperties.getManageUserId(),
                    GroupMemberTypeEnum.MANAGE);

            getGroupMemberService().addMember(imGroupEntity.getId(),
                    imGroupEntity.getName(),
                    redPacketProperties.getChatRoomTimingRedPacketAssistantUserId(),
                    GroupMemberTypeEnum.ACTIVITY);

            int randomNum = RandomUtils.nextInt(imGroupEntity.getMinVirtualNum(), imGroupEntity.getMaxVirtualNum());
            message = getGroupMemberService().addVirtualMember(imGroupEntity.getId(), imGroupEntity.getName(), randomNum);
        }

        groupActionService.saveHistory(GroupActionEnum.CREATE, imGroupEntity.getId(), null, null);

        return message;
    }

    /**
     * 初始化群组内管理员
     */
    private void initGroupManageUser() {
        UserInfoBean userInfo = userIntegrationService.getUserInfo(groupProperties.getManageUserId());
        imsdkAdapter.update(userInfo);

        UserInfoBean redpacketUser = userIntegrationService.getUserInfo(redPacketProperties.getChatRoomTimingRedPacketAssistantUserId());
        imsdkAdapter.update(redpacketUser);
    }

    private Message afterUpdate(ImGroupEntity oldEntity, ImGroupEntity imGroupEntity) {
        Message message = Message.build();
        // 重置参与人数
        if (!Objects.equals(oldEntity.getMemberMaxNum(), imGroupEntity.getMemberMaxNum())) {
            groupTotalService.resetMaxMember(imGroupEntity.getId(), imGroupEntity.getMemberMaxNum());
        }
        // 刷新标题
        if (!StringUtils.equals(oldEntity.getName(), imGroupEntity.getName())) {
            message.append(imsdkAdapter.refreshGroup(imGroupEntity.getId(), imGroupEntity.getName()));
        }

        // 获取群内参与的虚拟用户总数，如果大于当前上限，则扣除一部分，如果小于下限，则增加一部分
        int virtualUserTotal = imGroupMemberMapper.countByUserType(oldEntity.getId(), GroupMemberTypeEnum.VIRTUAL.getCode());

        if (virtualUserTotal > imGroupEntity.getMaxVirtualNum()) {
            List<ImGroupMemberEntity> removeUserList = imGroupMemberMapper.queryMember(oldEntity.getId(),
                    GroupMemberTypeEnum.VIRTUAL.getCode(),
                    groupProperties.getManageUserId(),
                    virtualUserTotal - imGroupEntity.getMaxVirtualNum()
            );

            for (ImGroupMemberEntity memberEntity : removeUserList) {
                imGroupMemberMapper.deleteById(memberEntity.getId());
            }
        } else if (virtualUserTotal < imGroupEntity.getMinVirtualNum()) {
            int randomNum = RandomUtils.nextInt(imGroupEntity.getMinVirtualNum(), imGroupEntity.getMaxVirtualNum());
            message = getGroupMemberService().addVirtualMember(imGroupEntity.getId(),
                    imGroupEntity.getName(),
                    randomNum - virtualUserTotal);
        }

        groupActionService.saveHistory(GroupActionEnum.MODIFY, imGroupEntity.getId(), null, null);

        reloadRuntimeCache(imGroupEntity.getId());
        return message;
    }

    @Override
    public Message complain(ComplainGroupParam param) {
        ImGroupComplainEntity entity = new ImGroupComplainEntity();
        entity.setId(SequenceHolder.nextLongId());
        entity.setGroupId(param.getGroupId());
        entity.setContent(param.getContent());
        entity.setCreateTime(new Date());
        entity.setUserId(param.getUserId());

        return Message.build(imGroupComplainMapper.insert(entity));
    }

    @Override
    public ImGroupEntity getGroupEntity(Long groupId) {
        Preconditions.checkNotNull(groupId);

        return imGroupMapper.selectById(groupId);
    }

    @Override
    public GroupRuntimeInfoDTO loadGroupRuntimeInfo(Long groupId) {
        GroupRuntimeInfoDTO runtimeInfoDTO = redisStringAdapter.get(GROUP_RUNTIME_CACHE_KEY.copy().appendKey(groupId), GroupRuntimeInfoDTO.class);

        if (null == runtimeInfoDTO) {
            runtimeInfoDTO = loadToCache(groupId);
        }
        return runtimeInfoDTO;
    }

    private GroupRuntimeInfoDTO loadToCache(Long groupId) {
        GroupRuntimeInfoDTO runtimeInfoDTO = imGroupMapper.queryRuntimeInfo(groupId);

        // 获取群内真实用户总数
        int currentTrueNum = imGroupMemberMapper.countByUserType(groupId, GroupMemberTypeEnum.NORMAL.getCode());
        // 如果群组内的当前用户达到参与人数，则不继续构建群组的显示图片，有2个是特殊的人
        if (currentTrueNum < 12 && runtimeInfoDTO.getCurrentNum() < runtimeInfoDTO.getTotalNum()) {
            groupMemberHeadImgService.groupHeadImgSplicing(groupId);
        }
        redisStringAdapter.set(GROUP_RUNTIME_CACHE_KEY.copy().appendKey(groupId), runtimeInfoDTO);

        return runtimeInfoDTO;
    }

    @Override
    public GroupRuntimeInfoDTO reloadRuntimeCache(Long groupId) {
        redisStringAdapter.remove(GROUP_RUNTIME_CACHE_KEY.copy().appendKey(groupId));
        return loadToCache(groupId);
    }
}
