package com.bxm.localnews.im.thirdpart.impl;

import com.bxm.egg.message.vo.UserInfoBean;
import com.bxm.localnews.im.config.IMProperties;
import com.bxm.localnews.im.param.MessageWithdrawParam;
import com.bxm.localnews.im.thirdpart.IMSDKAdapter;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.DateUtils;
import io.rong.RongCloud;
import io.rong.messages.BaseMessage;
import io.rong.models.Result;
import io.rong.models.group.GroupMember;
import io.rong.models.group.GroupModel;
import io.rong.models.message.GroupMessage;
import io.rong.models.message.PrivateMessage;
import io.rong.models.message.RecallMessage;
import io.rong.models.response.TokenResult;
import io.rong.models.user.UserModel;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.ApplicationArguments;
import org.springframework.boot.ApplicationRunner;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import java.util.Date;

@Component
@Slf4j
public class RongCloudIMSDK implements IMSDKAdapter, ApplicationRunner {

    private RongCloud rongCloud;

    private final IMProperties imProperties;

    /**
     * 禁言的最长时间
     */
    private final static int MAX_MUTE_MINUTES = 43200;

    @Autowired
    public RongCloudIMSDK(IMProperties imProperties) {
        this.imProperties = imProperties;
    }

    @Override
    public String token(UserInfoBean user) {
        UserModel model = build(user);

        TokenResult token = applyWithResult(() -> rongCloud.user.register(model));
        if (token == null) {
            return null;
        }

        return token.getToken();
    }

    private UserModel build(UserInfoBean user) {
        return new UserModel(user.getId().toString(), user.getNickname(), user.getHeadImg());
    }

    @Override
    public void update(UserInfoBean user) {
        UserModel model = build(user);
        if (log.isDebugEnabled()) {
            log.debug("用户信息变更，同步变更信息,用户信息：{}", model);
        }
        apply(() -> rongCloud.user.update(model));
    }

    @Override
    public Message block(Long userId, int minutes) {
        UserModel user = new UserModel();
        user.setMinute(minutes);
        user.setId(String.valueOf(userId));

        return apply(() -> rongCloud.user.block.add(user));
    }

    @Override
    public Message unblock(Long userId) {
        return apply(() -> rongCloud.user.block.remove(String.valueOf(userId)));
    }

    @Override
    public Message messageWithdraw(MessageWithdrawParam param) {
        RecallMessage recallMessage = new RecallMessage()
                .setSenderId(String.valueOf(param.getFromUserId()))
                .setTargetId(param.getTargetId())
                .setuId(param.getMessageUID())
                .setSentTime(String.valueOf(System.currentTimeMillis()));
        recallMessage.setIsAdmin(1);

        return apply(() -> rongCloud.message.chatroom.recall(recallMessage));
    }

    @Override
    public Message batchSendPersonalMessage(PrivateMessage message) {
        return apply(() -> rongCloud.message.msgPrivate.send(message));
    }

    @Override
    public Message createGroup(Long groupId, String name, Long... manageMember) {
        GroupModel groupModel = new GroupModel();
        groupModel.setId(groupId.toString());
        groupModel.setName(name);
        groupModel.setMembers(buildGroupMember(groupId, manageMember));

        return apply(() -> rongCloud.group.create(groupModel));
    }

    @Override
    public Message refreshGroup(Long groupId, String name) {
        GroupModel groupModel = new GroupModel();
        groupModel.setId(groupId.toString());
        groupModel.setName(name);

        return apply(() -> rongCloud.group.update(groupModel));
    }

    @Override
    public Message joinGroup(Long groupId, String groupName, Long... memberIds) {
        GroupModel groupModel = new GroupModel();
        groupModel.setId(groupId.toString());
        groupModel.setName(groupName);
        groupModel.setMembers(buildGroupMember(groupId, memberIds));

        return apply(() -> rongCloud.group.join(groupModel));
    }

    private GroupMember[] buildGroupMember(Long groupId, Long... memberIds) {
        GroupMember[] memberArray = new GroupMember[memberIds.length];
        for (int i = 0; i < memberIds.length; i++) {
            memberArray[i] = new GroupMember(memberIds[0].toString(), groupId.toString(), 0);
        }
        return memberArray;
    }

    @Override
    public Message quitGroup(Long groupId, Long... memberIds) {
        GroupModel groupModel = new GroupModel();
        groupModel.setId(groupId.toString());
        groupModel.setMembers(buildGroupMember(groupId, memberIds));

        return apply(() -> rongCloud.group.quit(groupModel));
    }

    @Override
    public Message blockGroupMember(Long groupId, Long memberId, Date blockDate) {
        long minutes = DateUtils.getDiffMinutes(blockDate, new Date(), true);
        if (minutes > MAX_MUTE_MINUTES) {
            minutes = 0L;
        }

        if (log.isDebugEnabled()) {
            log.debug("群成员禁言，群组：{},memberId: {} ,blockMinutes: {}", groupId, memberId, minutes);
        }

        GroupMember member = new GroupMember();
        member.setGroupId(groupId.toString());
        member.setId(memberId.toString());
        member.setMunite((int) minutes);

        GroupModel groupModel = new GroupModel();
        groupModel.setId(groupId.toString());
        groupModel.setMembers(new GroupMember[]{member});
        groupModel.setMinute((int) minutes);

        return apply(() -> rongCloud.group.muteMembers.add(groupModel));
    }

    @Override
    public Message unblockGroupMember(Long groupId, Long memberId) {
        if (log.isDebugEnabled()) {
            log.debug("解除群组禁言：{} - {} ", groupId, memberId);
        }
        GroupModel groupModel = new GroupModel();
        groupModel.setId(groupId.toString());
        groupModel.setMembers(buildGroupMember(groupId, memberId));

        return apply(() -> rongCloud.group.muteMembers.remove(groupModel));
    }

    @Override
    public Message sendGroupMessage(Long groupId, Long forumId, BaseMessage baseMessage) {
        if (log.isDebugEnabled()) {
            log.debug("发送消息到群组[{}],发送人:[{}],消息内容：[{}]",
                    groupId,
                    forumId,
                    JSON.toJSONString(baseMessage));
        }

        GroupMessage message = new GroupMessage();

        message.setContent(baseMessage);
        message.setObjectName(message.getContent().getType());
        message.setSenderId(forumId.toString());
        message.setTargetId(new String[]{groupId.toString()});

        return apply(() -> rongCloud.message.group.send(message));
    }

    /**
     * 执行融云的远程调用
     *
     * @param function 函数封装
     * @return 返回结果
     */
    private Message apply(ExecuteRongCloudFunction<Result> function) {
        Message message = Message.build();
        try {
            Result result = function.apply();

            if (null != result && HttpStatus.OK.value() == result.getCode()) {
                return message;
            }

            log.error("执行融云的远程调用失败，返回参数：{}", JSON.toJSONString(result));
            message.setSuccess(false).setMessage("调用融云的服务失败");
        } catch (Exception e) {
            log.error(e.getMessage(), e);
            message = Message.build(false, e.getMessage());
        }

        return message;
    }

    private <T> T applyWithResult(ExecuteRongCloudFunction<T> function) {
        try {
            return function.apply();
        } catch (Exception e) {
            log.error(e.getMessage(), e);
        }
        return null;
    }

    @FunctionalInterface
    interface ExecuteRongCloudFunction<T> {
        /**
         * 执行融云的SDK调用
         *
         * @return 融云调用结果
         * @throws Exception 调用过程中发生异常
         */
        T apply() throws Exception;
    }

    @Override
    public void run(ApplicationArguments args) throws Exception {
        if (null != imProperties.getRongcloudAppKey() && null != imProperties.getRongcloudAppSecret()) {
            rongCloud = RongCloud.getInstance(imProperties.getRongcloudAppKey(), imProperties.getRongcloudAppSecret());
        } else {
            log.error("未配置融云的相关信息");
        }
    }

}
