package com.bxm.localnews.im.user.impl;

import com.bxm.egg.common.constant.InfoCombineStateEnum;
import com.bxm.egg.message.integration.UserIntegrationService;
import com.bxm.localnews.im.config.IMProperties;
import com.bxm.localnews.im.domain.ChatMessageMapper;
import com.bxm.localnews.im.dto.SessionInfoDto;
import com.bxm.localnews.im.user.UserService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.Objects;

import static com.bxm.localnews.im.constant.ImRedisKey.SHOW_INFO_COMPLETE_GUIDE_FLAG;
import static com.bxm.localnews.im.constant.ImRedisKey.UN_READ_MSG;

@Service
@Slf4j
public class UserServiceImpl implements UserService {

    private final UserIntegrationService userIntegrationService;

    private final ChatMessageMapper chatMessageMapper;

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final IMProperties imProperties;

    private final RedisSetAdapter redisSetAdapter;

    /**
     * 用户未回复的情况下，最多发送的消息数量
     * 因客户端会在发送后立即请求，如限制为3条，则最多数量为2
     */
    private final static int MAX_SEND_MSG = 2;

    @Autowired
    public UserServiceImpl(UserIntegrationService userIntegrationService,
                           ChatMessageMapper chatMessageMapper,
                           RedisHashMapAdapter redisHashMapAdapter,
                           IMProperties imProperties,
                           RedisSetAdapter redisSetAdapter) {
        this.userIntegrationService = userIntegrationService;
        this.chatMessageMapper = chatMessageMapper;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.imProperties = imProperties;
        this.redisSetAdapter = redisSetAdapter;
    }

    @Override
    public SessionInfoDto get(Long fromUid, Long toUid, Boolean consumePromptGuide) {
        SessionInfoDto session = new SessionInfoDto();
        //用户是否关注对方
        session.setFollow(userIntegrationService.isFollow(fromUid, toUid));
        //对方是否关注我
        session.setFollowMe(userIntegrationService.isFollow(toUid, fromUid));

        //获取聊天目标用户信息
        session.setTargetUser(userIntegrationService.getUserInfo(toUid));
        //判断用户是否可发送信息给对方
        session.setAllowSend(true);

        //提供白名单过滤，为运营提供绿色通道（2019-12-21 23点.....）
        if (imProperties.getWhiteList().contains(fromUid)) {
            session.setAllowSendMedia(true);
        } else {
            //对方是否关注当前用户
            if (session.getFollowMe()) {
                session.setFollowMsg(userIntegrationService.hasFollowMsg(fromUid, toUid));
                //对方有关注我，则允许发送多媒体文件
                session.setAllowSendMedia(true);
            } else {
                //对方是否给当前用户进行回复
                Integer reply = chatMessageMapper.hasReply(fromUid, toUid);
                if (null == reply || 0 == reply) {
                    //对方的未读消息中，当前用户的发送量（即单方面发送的消息数量）
                    Long sendMsgNum = redisHashMapAdapter.get(buildKey(toUid), String.valueOf(fromUid), Long.class);
                    if (null != sendMsgNum && sendMsgNum >= MAX_SEND_MSG) {
                        session.setAllowSend(false);
                        session.setRejectMsg("对方还没有关注你，在未收到回复前不能再继续发消息了");
                        log.debug("session:{},sendMsgNum:{}", session, sendMsgNum);
                    }
                } else {
                    //对方有回复，则可发送多媒体文件
                    session.setAllowSendMedia(true);
                }
            }
        }

        //是否消费引导完善资料提示
        if (Objects.nonNull(consumePromptGuide) && consumePromptGuide) {
            //根据用户的完善资料度判断是否发送引导完善资料的提示
            String value = Long.toHexString(fromUid) + Long.toHexString(toUid);
            //判断用户有没有和目标用户聊天时发过引导资料的提示
            if (!redisSetAdapter.exists(SHOW_INFO_COMPLETE_GUIDE_FLAG.copy(), value)) {
                Long infoCompleteState = userIntegrationService.getInfoCompleteState(fromUid);
                if (Objects.nonNull(infoCompleteState)) {
                    int infoCompletePercent = InfoCombineStateEnum.getInfoCompletePercent(infoCompleteState);
                    //如果完善度小于80，则提示用户显示引导资料完善的提示
                    if (imProperties.getNativeNewbieGuideThreshold() > infoCompletePercent) {
                        session.setShowInfoCompleteGuideFlag(true);
                        redisSetAdapter.add(SHOW_INFO_COMPLETE_GUIDE_FLAG.copy(), value);
                    }
                }
            }
        }

        return session;
    }

    private KeyGenerator buildKey(Long userId) {
        return UN_READ_MSG.copy().appendKey(userId);
    }
}
