package com.bxm.egg.user.login.impl;

import com.bxm.egg.mq.common.constant.SmsTemplateEnum;
import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.integration.SmsIntegrationService;
import com.bxm.egg.user.login.UserLoginService;
import com.bxm.egg.user.login.strategy.AbstractUserLoginStrategy;
import com.bxm.egg.user.model.dto.login.LoginResultDTO;
import com.bxm.egg.user.model.dto.token.TokenInfoDTO;
import com.bxm.egg.user.model.param.login.LoginParam;
import com.bxm.egg.user.model.param.login.LogoutParam;
import com.bxm.egg.user.model.param.login.UserSettingsPasswordParam;
import com.bxm.egg.user.token.TokenService;
import com.bxm.newidea.component.bo.Message;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import java.util.List;
import java.util.Map;

/**
 * 用户登录service实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/14 10:24 上午
 */
@Slf4j
@Service
public class UserLoginServiceImpl implements UserLoginService {

    private Map<UserLoginTypeEnum, AbstractUserLoginStrategy> loginStrategyMap;

    private final TokenService tokenService;

    private final UserInfoService userInfoService;

    private final SmsIntegrationService smsIntegrationService;

    public UserLoginServiceImpl(List<AbstractUserLoginStrategy> loginStrategys,
                                TokenService tokenService,
                                UserInfoService userInfoService,
                                SmsIntegrationService smsIntegrationService) {
        this.tokenService = tokenService;
        this.userInfoService = userInfoService;
        this.smsIntegrationService = smsIntegrationService;

        loginStrategyMap = Maps.newHashMap();

        for (AbstractUserLoginStrategy loginStrategy : loginStrategys) {
            loginStrategyMap.put(loginStrategy.type(), loginStrategy);
        }

    }

    @Override
    public <T extends LoginParam> Message excLogin(T loginParam, UserLoginTypeEnum loginType) {

        Message message = loginStrategyMap.get(loginType).doLogin(loginParam);

        if (!message.isSuccess()) {
            return message;
        }

        LoginResultDTO loginResultDTO = message.getParam("loginResultDTO");

        if (loginResultDTO == null) {
            return Message.build(false, "账户存在异常，无法登录");
        }

        TokenInfoDTO tokenInfoDTO = tokenService.create(loginResultDTO.getUserId(), loginParam);
        loginResultDTO.setAccessToken(tokenInfoDTO.getAccessToken());
        loginResultDTO.setRefreshToken(tokenInfoDTO.getRefreshToken());

        Message resultMsg = Message.build(true);
        resultMsg.addParam("loginResultDTO", loginResultDTO);

        return resultMsg;
    }

    @Override
    public void exitLogin(LogoutParam logoutParam) {
        //退出登录暂不实现
    }

    @Override
    public Message settingsPassword(UserSettingsPasswordParam userSettingsPasswordParam) {

        //判断验证码是否正确
        if (!smsIntegrationService.verifyByType(SmsTemplateEnum.RESET_PWD,
                userSettingsPasswordParam.getPhone(),
                userSettingsPasswordParam.getSmsCode())) {
            return Message.build(false, "验证码错误");
        }

        //修改密码
        userInfoService.updateUserPassword(userSettingsPasswordParam.getUserId(),
                userSettingsPasswordParam.getPassword());

        return Message.build(true);
    }


}