package com.bxm.egg.user.medal.cache;

import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.mapper.medal.UserMedalInfoMapper;
import com.bxm.egg.user.model.entity.medal.UserMedalInfoEntity;
import com.bxm.newidea.component.redis.RedisListAdapter;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.stereotype.Service;

import java.util.*;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 成就勋章缓存管理器
 *
 * @author wzy
 * @version 1.0
 * @date 2021/3/4 3:37 下午
 */
@Slf4j
@Service
@AllArgsConstructor
public class AchievementMedalCacheManager {

    private RedisListAdapter redisListAdapter;

    private UserMedalInfoMapper userMedalInfoMapper;

    /**
     * 获取所有的成就勋章缓存
     *
     * @return 成就勋章缓存列表
     */
    public List<UserMedalInfoEntity> getAllAchievementMedalList() {
        List<UserMedalInfoEntity> achievementMedalList = redisListAdapter.range(RedisConfig.ACHIEVEMNET_MEDAL_DATA.copy(), 0, -1, UserMedalInfoEntity.class);
        if (CollectionUtils.isEmpty(achievementMedalList)) {
            if (log.isDebugEnabled()) {
                log.debug("成就勋章缓存失效，重新获取");
            }
            List<UserMedalInfoEntity> dbAchievementMedalList = userMedalInfoMapper.getAllAchievementMedalInfo();
            achievementMedalList = dbAchievementMedalList;
            redisListAdapter.leftPush(RedisConfig.ACHIEVEMNET_MEDAL_DATA.copy(), dbAchievementMedalList);
        }

        return achievementMedalList;
    }

    /**
     * 获取勋章ID对应勋章信息的map
     *
     * @return map信息
     */
    public Map<Long, UserMedalInfoEntity> getMedalId2InfoMap() {

        List<UserMedalInfoEntity> allAchievementMedalList = getAllAchievementMedalList();

        return allAchievementMedalList.stream().filter(userMedalInfoEntity -> Objects.nonNull(userMedalInfoEntity.getParentId()))
                .collect(Collectors.toMap(UserMedalInfoEntity::getId, Function.identity()));
    }

    /**
     * 返回父勋章id对应子勋章信息列表的map
     *
     * @return map信息
     */
    public Map<Long, List<UserMedalInfoEntity>> getMedalParentId2InfoListMap() {

        Map<Long, List<UserMedalInfoEntity>> userMedalMap = new HashMap<>();

        List<UserMedalInfoEntity> allAchievementMedalList = getAllAchievementMedalList();

        List<Long> medalParentIdList = allAchievementMedalList.stream().filter(userMedalInfoEntity -> Objects.isNull(userMedalInfoEntity.getParentId()))
                .map(UserMedalInfoEntity::getId)
                .collect(Collectors.toList());

        for (Long medalParentId : medalParentIdList) {
            List<UserMedalInfoEntity> medalInfoEntityList = allAchievementMedalList.stream()
                    .filter(medalItem -> Objects.nonNull(medalItem.getParentId()) && medalItem.getParentId().equals(medalParentId))
                    .collect(Collectors.toList());
            userMedalMap.put(medalParentId, medalInfoEntityList);
        }

        return userMedalMap;
    }

    /**
     * 返回父勋章id对应子勋章信息列表
     *
     * @return map信息
     */
    public List<Long> getAllMedalParentIdList() {

        List<UserMedalInfoEntity> allAchievementMedalList = getAllAchievementMedalList();

        return allAchievementMedalList.stream().filter(userMedalInfoEntity -> Objects.isNull(userMedalInfoEntity.getParentId()))
                .map(UserMedalInfoEntity::getId)
                .collect(Collectors.toList());
    }

    /**
     * 判断当前等级是否是该勋章的最高等级
     *
     * @param parentMedalId 父勋章id
     * @param currentLevel  当前等级
     * @return 是否是最高等级
     */
    public boolean isMaxLevel(Long parentMedalId, Integer currentLevel) {

        Map<Long, List<UserMedalInfoEntity>> medalParentId2InfoListMap = getMedalParentId2InfoListMap();

        List<UserMedalInfoEntity> userMedalInfoEntities = medalParentId2InfoListMap.get(parentMedalId);

        for (UserMedalInfoEntity userMedalInfoEntity : userMedalInfoEntities) {
            if (userMedalInfoEntity.getLevel() > currentLevel) {
                return false;
            }
        }

        return true;
    }

    /**
     * 获取每种勋章类型->对应具体子勋章类型列表（经过排序，等级从小到大）
     *
     * @return 勋章列表信息
     */
    public List<UserMedalInfoEntity> getMedalListByType(String medalType) {

        List<UserMedalInfoEntity> allAchievementMedalList = getAllAchievementMedalList();

        Map<String, List<UserMedalInfoEntity>> resultMap = allAchievementMedalList.stream().filter(achievementMedal -> Objects.nonNull(achievementMedal.getParentId()))
                .collect(Collectors.groupingBy(UserMedalInfoEntity::getType));

        resultMap.forEach((type, entityList) -> entityList.sort(Comparator.comparing(UserMedalInfoEntity::getLevel)));


        return resultMap.get(medalType);
    }
}