package com.bxm.egg.user.follow.impl;

import com.alibaba.fastjson.JSON;
import com.baomidou.mybatisplus.core.metadata.IPage;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import com.bxm.component.mybatis.dto.PlusPageModelDTO;
import com.bxm.egg.common.enums.UserStatusEnum;
import com.bxm.egg.user.attribute.UserAttributeService;
import com.bxm.egg.user.attribute.UserFunsService;
import com.bxm.egg.user.attribute.UserTagService;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.enums.UserFollowStatusEnum;
import com.bxm.egg.user.equitylevelmedal.impl.UserEquityLevelMedalServiceImpl;
import com.bxm.egg.user.follow.UserFollowService;
import com.bxm.egg.user.info.UserInfoCacheService;
import com.bxm.egg.user.integration.MessageFacadeIntegrationService;
import com.bxm.egg.user.integration.NewsIntegrationService;
import com.bxm.egg.user.integration.UserSyncIntegrationService;
import com.bxm.egg.user.integration.sync.SixEnjoyFriendsIntegrationService;
import com.bxm.egg.user.mapper.UserFollowMapper;
import com.bxm.egg.user.mapper.UserFunsMapper;
import com.bxm.egg.user.model.UserEquityDTO;
import com.bxm.egg.user.model.bo.UserCacheInfoBO;
import com.bxm.egg.user.model.dto.warmlevel.UserWarmLevelInfoDTO;
import com.bxm.egg.user.model.param.UserFollowParam;
import com.bxm.egg.user.model.vo.UserFollow;
import com.bxm.egg.user.model.vo.UserFollowRecord;
import com.bxm.egg.user.properties.NativeUserProperties;
import com.bxm.egg.user.warmlevel.UserWarmLevelService;
import com.bxm.newidea.component.dto.IPageModel;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisSetAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.*;
import java.util.stream.Collectors;

import static java.lang.Boolean.TRUE;

@Slf4j
@Service
public class UserFollowServiceImpl implements UserFollowService {

    private static final String USER_FOLLOW = "USER_FOLLOW_";

    /**
     * 关注缓存消息过期时间
     */
    private static final Long FOLLOW_CACHE_EXPIRED = 60 * 60 * 24 * 15L;

    @Resource
    private UserFollowMapper userFollowMapper;

    @Resource
    private UserFunsMapper userFunsMapper;

    @Resource
    private UserAttributeService userAttributeService;

    @Resource
    private RedisSetAdapter redisSetAdapter;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private DistributedLock distributedLock;

    @Resource
    private UserFunsService userFunsService;

    @Resource
    private MessageFacadeIntegrationService messageFacadeIntegrationService;

    @Resource
    private NativeUserProperties nativeUserProperties;

    @Resource
    private NewsIntegrationService newsIntegrationService;

    @Resource
    private UserTagService userTagService;

    @Autowired
    private UserWarmLevelService userWarmLevelService;

    @Autowired
    private UserEquityLevelMedalServiceImpl userEquityLevelMedalService;

    @Resource
    private UserInfoCacheService userInfoCacheService;

    @Resource
    private UserSyncIntegrationService userSyncIntegrationService;

    @Resource
    private SixEnjoyFriendsIntegrationService sixEnjoyFriendsIntegrationService;

    @Override
    public Boolean isFollowed(Long currentUserId, Long targetUserId) {
        KeyGenerator keyGenerator = getFollowRedisKey(currentUserId);
        if (!redisSetAdapter.hasKey(keyGenerator)) {
            //从数据库中取
            List<Long> targetUserIdList = userFollowMapper.getFollowedUserIdList(currentUserId);
            if (CollectionUtils.isEmpty(targetUserIdList)) {
                //查询为空时，放入特殊值，防止后续继续查询数据库
                redisSetAdapter.add(keyGenerator, -1L);
                return false;
            }
            redisSetAdapter.add(keyGenerator, targetUserIdList.toArray());
        }

        return redisSetAdapter.exists(keyGenerator, targetUserId);
    }

    @Override
    public List<Long> isFolloweds(Long currentUserId, List<Long> targetUserIds) {
        List<Long> result = new ArrayList<>();
        for (Long targetUserId : targetUserIds) {
            if (this.isFollowed(currentUserId, targetUserId)) {
                result.add(targetUserId);
            }
        }
        return result;
    }

    @Override
    public Boolean hasFollowMsg(Long userId, Long targetUserId) {
        //是否存在目标用户曾经关注当前用户的记录
        KeyGenerator key = RedisConfig.COMSUME_FOLLOW_INFO.copy().appendKey(userId).appendKey(targetUserId);

        if (redisStringAdapter.hasKey(key)) {
            redisStringAdapter.remove(key);
            return true;
        }

        return false;
    }

    @Override
    public Boolean follow(Long userId, Long followUserId, Byte type) {
        if (notValidUser(userId) || notValidUser(followUserId)) {
            return false;
        }

        if (Objects.equals(userId, followUserId)) {
            log.warn("用户: {} 自己关注自己", userId);
            return true;
        }

        String lockKey = USER_FOLLOW + userId + "_" + followUserId;

        //关注或者取消关注，需要使用分布式锁，需注意幂等操作
        if (distributedLock.lock(lockKey)) {
            //当前用户是否关注对方
            boolean currentFollow = isFollowed(userId, followUserId);
            //对方是否关注当前用户
            boolean targetFollow = isFollowed(followUserId, userId);

            //如果是关注，并且当前用户未关注目标用户(已关注则不进行处理)
            if (UserFollowStatusEnum.FOLLOW.getCode() == type && !currentFollow) {
                //如果目标用户也关注了当前用户，则更新对方的记录为相互关注
                if (targetFollow) {
                    //增加当前用户相互关注的记录
                    addFollow(userId, followUserId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                    //对方用户的关注变成相互关注
                    modifyFollow(followUserId, userId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                    //从当前用户的粉丝中设置为相互关注
                    userFunsService.setFuns(userId, followUserId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                    //对方用户的粉丝新增互相关注
                    userFunsService.addFuns(followUserId, userId, UserFollowStatusEnum.EACH_FOLLOW.getCode());

                    //添加蛋蛋佳好友关系
                    sixEnjoyFriendsIntegrationService.insertFriendsToSixEnjoy(userId, followUserId);

                } else {
                    //增加一条关注记录
                    addFollow(userId, followUserId, UserFollowStatusEnum.FOLLOW.getCode());
                    //给对方增加粉丝
                    userFunsService.addFuns(followUserId, userId, UserFollowStatusEnum.FOLLOW.getCode());
                }

                //更新用户的关注、粉丝信息
                userAttributeService.addUserFollowCount(userId, followUserId, true);
                //关注成功
                messageFacadeIntegrationService.addFollowMessage(followUserId);
                //通知新闻服务关注
                userSyncIntegrationService.addFollow(userId, followUserId);
            } else if (UserFollowStatusEnum.UNFOLLOW.getCode() == type && currentFollow) {
                //将当前用户从对方的粉丝中移除
                userFunsService.removeFuns(followUserId, userId);
                //删除当前用户的关注列表
                removeFollow(userId, followUserId);
                //如果对方关注了当前用户
                if (targetFollow) {
                    //更新对方的关注状态为关注
                    modifyFollow(followUserId, userId, UserFollowStatusEnum.FOLLOW.getCode());
                    //将当前用户粉丝中的目标用户更新为关注
                    userFunsService.setFuns(userId, followUserId, UserFollowStatusEnum.FOLLOW.getCode());
                }

                //扣除用户的关注、目标用户的粉丝数
                userAttributeService.addUserFollowCount(userId, followUserId, false);
                //通知新闻服务取消关注
                userSyncIntegrationService.removeFollow(userId, followUserId);
                //解除蛋蛋佳好友关系
                sixEnjoyFriendsIntegrationService.delFriendsToSixEnjoy(userId, followUserId);
            }
            //解除分布式锁
            distributedLock.unlock(lockKey);
        }

        return true;
    }

    /**
     * 增加一条关注记录
     *
     * @param userId   用户ID
     * @param followId 关注目标用户ID
     * @param type     关注类型
     */
    private void addFollow(Long userId, Long followId, byte type) {
        UserFollowRecord follow = new UserFollowRecord();
        follow.setUserId(userId);
        follow.setFollowedUserId(followId);
        follow.setType(type);

        int update = userFollowMapper.updateFollowedStatus(follow);
        if (0 == update) {
            follow.setId(SequenceHolder.nextLongId());
            userFollowMapper.insertFollowed(follow);
        }
        redisSetAdapter.add(getFollowRedisKey(userId), followId);

        //给被关注人增加一条消费消息
        KeyGenerator comsumeKey = RedisConfig.COMSUME_FOLLOW_INFO.copy().appendKey(followId).appendKey(userId);
        redisStringAdapter.set(comsumeKey, StringUtils.EMPTY, FOLLOW_CACHE_EXPIRED);
    }

    /**
     * 变更关注记录类型
     *
     * @param userId   用户ID
     * @param followId 关注目标用户ID
     * @param type     关注类型
     */
    private void modifyFollow(Long userId, Long followId, byte type) {
        if (UserFollowStatusEnum.UNFOLLOW.getCode() == type) {
            removeFollow(userId, followId);
        } else {
            UserFollowRecord follow = new UserFollowRecord();
            follow.setUserId(userId);
            follow.setFollowedUserId(followId);
            follow.setType(type);

            userFollowMapper.updateFollowedStatus(follow);
        }
    }

    /**
     * 移除关注记录
     *
     * @param userId   用户ID
     * @param followId 关注目标用户ID
     */
    private void removeFollow(Long userId, Long followId) {
        UserFollowRecord follow = new UserFollowRecord();
        follow.setUserId(userId);
        follow.setFollowedUserId(followId);
        follow.setType(UserFollowStatusEnum.UNFOLLOW.getCode());

        userFollowMapper.updateFollowedStatus(follow);
        redisSetAdapter.remove(getFollowRedisKey(userId), followId);
    }

    @Override
    public IPageModel<UserFollow> followList(UserFollowParam param) {

        IPage<UserFollow> queryParam = new Page<>(param.getPageNum(), param.getPageSize());
        IPage<UserFollow> pageResult = userFollowMapper.queryFollowByPage(queryParam, param);

        fillExtraInfo(pageResult.getRecords());

        return PlusPageModelDTO.build(pageResult);
    }

    @Override
    public IPageModel<UserFollow> queryFunsByPage(UserFollowParam param) {

        IPage<UserFollow> queryParam = new Page<>(param.getPageNum(), param.getPageSize());
        IPage<UserFollow> pageResult = userFunsMapper.queryFunsByPage(queryParam, param);

        fillExtraInfo(pageResult.getRecords());

        for (UserFollow record : pageResult.getRecords()) {
            //特殊处理一下，对于粉丝来说，0这个关注状态其实表示的就是未关注，2表示的相互关注，只有这两种情况
            //这里特殊处理将0转成1，就是未关注
            if (Objects.equals(record.getStatus(), UserFollowStatusEnum.FOLLOW.getCode())) {
                record.setStatus(UserFollowStatusEnum.UNFOLLOW.getCode());
            }
        }

        return PlusPageModelDTO.build(pageResult);
    }

    @Override
    public List<Long> followUserIdList(Long userId) {
        //缓存取
        Set<Long> list = redisSetAdapter.getAllMembers(getFollowRedisKey(userId), Long.class);
//        UserIdFollowParam param = new UserIdFollowParam();
//        param.setUserId(userId);
//        userFollowMapper.getFollowedUserIds(param);
        return new ArrayList<>(list);
    }

    private KeyGenerator getFollowRedisKey(Long userId) {
        return RedisConfig.USER_FOLLOW_LIST.copy().appendKey(userId);
    }

    private void fillExtraInfo(List<UserFollow> resultList) {
        if (!CollectionUtils.isEmpty(resultList)) {
            List<UserCacheInfoBO> userInfoList = batchLoadFormCache(resultList);

            resultList.forEach(userFollow -> {
                Optional<UserCacheInfoBO> userInfoOptional = userInfoList.stream()
                        .filter(user -> user.getUserId().equals(userFollow.getUserId()))
                        .findFirst();
                if (userInfoOptional.isPresent()) {
                    UserCacheInfoBO userInfo = userInfoOptional.get();
                    userFollow.setNickname(userInfo.getNickname());
                    userFollow.setHeadImg(userInfo.getHeadImg());
                    userFollow.setSex(userInfo.getSex());
                    if (log.isDebugEnabled()) {
                        log.debug("用户资料信息：{}.用户id：{}", JSON.toJSONString(userInfo.getUserInformationBO()),
                                userInfo.getUserId());
                    }

                    if (Objects.isNull(userInfo.getUserInformationBO()) || Objects.equals(userInfo.getUserInformationBO().getIsDefaultPersonalProfile(), TRUE)) {
                        userFollow.setPersonalProfile(nativeUserProperties.getDefaultPersonalProfile());
                    } else {
                        userFollow.setPersonalProfile(userInfo.getUserInformationBO().getPersonalProfile());
                    }
                }
                UserWarmLevelInfoDTO userWarmInfoCache = userWarmLevelService.getUserWarmInfoCache(userFollow.getUserId());
                UserEquityDTO equityDTO = userWarmInfoCache.getEquityDTO();
                userFollow.setUserWarmLevelDTO(equityDTO);
            });
        }
    }

    /**
     * 根据id列表批量取得用户的用户
     */
    private List<UserCacheInfoBO> batchLoadFormCache(List<UserFollow> resultList) {
        Set<Long> originUserIds = resultList.stream().map(UserFollow::getUserId).collect(Collectors.toSet());
        return userInfoCacheService.getBatchUserInfo(originUserIds);
    }

    /**
     * 无效的用户
     *
     * @param userId 用户id
     * @return 是否有效
     */
    private boolean notValidUser(Long userId) {
        UserCacheInfoBO userCacheInfoBO = userInfoCacheService.load(userId);
        return null == userCacheInfoBO ||
                Objects.equals(UserStatusEnum.DISABLE.getCode(),
                        userCacheInfoBO.getState());

    }

    @Override
    public void addHuolaFriends(Long userId, Long followUserId, Byte type) {
        //当前用户是否关注对方
        boolean currentFollow = isFollowed(userId, followUserId);
        //对方是否关注当前用户
        boolean targetFollow = isFollowed(followUserId, userId);

        //如果是关注，并且当前用户未关注目标用户(已关注则不进行处理)
        if (UserFollowStatusEnum.FOLLOW.getCode() == type &&!currentFollow) {
            //如果目标用户也关注了当前用户，则更新对方的记录为相互关注
            if (targetFollow) {
                //增加当前用户相互关注的记录
                addFollow(userId, followUserId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                //对方用户的关注变成相互关注
                modifyFollow(followUserId, userId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                //从当前用户的粉丝中设置为相互关注
                userFunsService.setFuns(userId, followUserId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
                //对方用户的粉丝新增互相关注
                userFunsService.addFuns(followUserId, userId, UserFollowStatusEnum.EACH_FOLLOW.getCode());
            } else {
                //增加一条关注记录
                addFollow(userId, followUserId, UserFollowStatusEnum.FOLLOW.getCode());
                //给对方增加粉丝
                userFunsService.addFuns(followUserId, userId, UserFollowStatusEnum.FOLLOW.getCode());
            }

            //更新用户的关注、粉丝信息
            userAttributeService.addUserFollowCount(userId, followUserId, true);
        } else if (UserFollowStatusEnum.UNFOLLOW.getCode() == type && currentFollow) {
            //将当前用户从对方的粉丝中移除
            userFunsService.removeFuns(followUserId, userId);
            //删除当前用户的关注列表
            removeFollow(userId, followUserId);
            //如果对方关注了当前用户
            if (targetFollow) {
                //更新对方的关注状态为关注
                modifyFollow(followUserId, userId, UserFollowStatusEnum.FOLLOW.getCode());
                //将当前用户粉丝中的目标用户更新为关注
                userFunsService.setFuns(userId, followUserId, UserFollowStatusEnum.FOLLOW.getCode());
            }
            //扣除用户的关注、目标用户的粉丝数
            userAttributeService.addUserFollowCount(userId, followUserId, false);
        }

    }
}
