package com.bxm.egg.user.info.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.egg.mq.common.constant.MessageTypeEnum;
import com.bxm.egg.user.account.UserAccountService;
import com.bxm.egg.user.attribute.UserVisitService;
import com.bxm.egg.user.constant.UserEggRoleEnum;
import com.bxm.egg.user.info.*;
import com.bxm.egg.user.integration.MessageFacadeIntegrationService;
import com.bxm.egg.user.integration.sync.SixEnjoyDataStatisticsIntegrationService;
import com.bxm.egg.user.integration.sync.SixEnjoyInviteFoodsIntegrationService;
import com.bxm.egg.user.invite.UserInviteService;
import com.bxm.egg.user.invite.egg.UserEggInviteService;
import com.bxm.egg.user.medal.UserMedalService;
import com.bxm.egg.user.model.UserEquityDTO;
import com.bxm.egg.user.model.bo.UserCacheInfoBO;
import com.bxm.egg.user.model.bo.UserInformationBO;
import com.bxm.egg.user.model.dto.UserBaseInfoDTO;
import com.bxm.egg.user.model.dto.homepage.*;
import com.bxm.egg.user.model.dto.info.UserBriefInfoDTO;
import com.bxm.egg.user.model.dto.info.UserDetailInfoDTO;
import com.bxm.egg.user.model.dto.medal.UserMedalDTO;
import com.bxm.egg.user.model.entity.UserAccountEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.entity.UserInformationEntity;
import com.bxm.egg.user.model.entity.UserStatisticsEntity;
import com.bxm.egg.user.properties.UserMedalProperties;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.egg.user.role.UserEggRoleService;
import com.bxm.egg.user.warmlevel.UserWarmLevelService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.sync.facade.dto.UserAssociateDataDTO;
import com.bxm.sync.facade.dto.UserCouponsStatisticsDTO;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.isBlank;

/**
 * 获取用户基本信息service实现类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/17 10:39 上午
 */
@Slf4j
@Service
@AllArgsConstructor
public class UserBaseInfoServiceImpl implements UserBaseInfoService {

    private final UserInfoCacheService userInfoCacheService;

    private final UserProperties userProperties;

    private final UserMedalProperties userMedalProperties;

    private final UserInfoService userInfoService;

    private final UserStatisticsService userStatisticsService;

    private final UserInformationService userInformationService;

    private final UserVisitService userVisitService;

    private final MessageFacadeIntegrationService messageFacadeIntegrationService;

    private final UserInviteService userInviteService;

    private final UserMedalService userMedalService;

    private final UserWarmLevelService userWarmLevelService;

    private final UserEggRoleService userEggRoleService;

    private final SixEnjoyDataStatisticsIntegrationService sixEnjoyDataStatisticsIntegrationService;

    private final UserAccountService userAccountService;

    private final UserEggInviteService userEggInviteService;

    private final SixEnjoyInviteFoodsIntegrationService sixEnjoyInviteFoodsIntegrationService;

    @Override
    public List<UserBriefInfoDTO> getUserBriefInfoList(List<Long> userIdList) {

        if (CollectionUtils.isEmpty(userIdList)) {
            return Lists.newArrayList();
        }

        return userIdList.stream()
                .map(this::getUserBriefInfo)
                .collect(Collectors.toList());
    }

    @Override
    public List<UserDetailInfoDTO> getUserDetailInfoList(List<Long> userIdList) {
        if (CollectionUtils.isEmpty(userIdList)) {
            return Lists.newArrayList();
        }

        return userIdList.stream()
                .map(this::getUserDetailInfo)
                .collect(Collectors.toList());
    }

    /**
     * 获取用户简略信息
     *
     * @param userId 用户id
     * @return 用户简略信息
     */
    private UserBriefInfoDTO getUserBriefInfo(Long userId) {
        UserBriefInfoDTO userBriefInfoDTO = new UserBriefInfoDTO();

        UserCacheInfoBO userCacheInfo = userInfoCacheService.load(userId);
        BeanUtils.copyProperties(userCacheInfo, userBriefInfoDTO);

        //填充是否是炫彩id
        UserEquityDTO userWarmLevelDTO = userCacheInfo.getUserWarmLevelDTO();
        boolean dazzling = userWarmLevelDTO != null
                && userWarmLevelDTO.getDazzling() != null
                && userWarmLevelDTO.getDazzling();

        userBriefInfoDTO.setDazzling(dazzling);

        return userBriefInfoDTO;

    }

    @Override
    public UserBaseInfoDTO getUserPersonalCenterInfo(Long userId) {
        //得到用户基本信息
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(userId);
        UserStatisticsEntity userStatisticsEntity = userStatisticsService.selectUserStatisticsByUserId(userId);
        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userId);
        UserAccountEntity userAccountInfo = userAccountService.getUserAccountInfo(userId);

        if (null == userInfoEntity) {
            return null;
        }
        UserBaseInfoDTO userBaseInfoDTO = new UserBaseInfoDTO();

        userBaseInfoDTO.setNickname(isBlank(userInfoEntity.getNickname()) ? StringUtils.hideMobile(userInfoEntity.getPhone())
                : userInfoEntity.getNickname());
        userBaseInfoDTO.setHeadImg(isBlank(userInfoEntity.getHeadImg()) ? userProperties.getDefaultHeadImageUrl() : userInfoEntity.getHeadImg());
        userBaseInfoDTO.setForumNum(userStatisticsEntity.getPostNum());

        userBaseInfoDTO.setSex(userInfoEntity.getSex());
        userBaseInfoDTO.setAttentionNum(userStatisticsEntity.getAttentionNum());
        userBaseInfoDTO.setFanNum(userStatisticsEntity.getFanNum());
        userBaseInfoDTO.setPersonalProfile(userInformationEntity.getPersonalProfile());
        userBaseInfoDTO.setAllInterviewNum(userVisitService.getInterviewNumByUserId(userId));

        userBaseInfoDTO.setUnReadFanNum(messageFacadeIntegrationService.getUnReadMessageNum(userId, MessageTypeEnum.ADD_FUNS));

        //今天增加的访问数量
        userBaseInfoDTO.setTodayInterviewNum(userVisitService.getTodayVisitorNum(userId));

        //用户最近的收徒列表
        userBaseInfoDTO.setInviteUserHeadImgList(userInviteService.getLastInviteUserHeadImgList(userId));

        //3.12.0 设置用户勋章个数信息
        int userMedalNum = userMedalService.getUserMedalNum(userId);

        UserMedalDTO userMedalDTO = new UserMedalDTO();
        userMedalDTO.setMedalIconUrl(userMedalProperties.getMyPageMedalUrl());
        userMedalDTO.setMedalNum(userMedalNum);

        userBaseInfoDTO.setUserMedalDTO(userMedalDTO);

        //3.12.0添加用户温暖值等级权益信息
        userBaseInfoDTO.setEquityDTO(userWarmLevelService.getUserWarmInfoCache(userId).getEquityDTO());

        //添加蛋蛋佳角色信息
        fillEggRoleInfo(userId, userBaseInfoDTO);

        //添加蛋蛋佳联营人信息
        fillEggAgentAndBusiness(userId, userBaseInfoDTO);

        //填充钱包信息
        fillEggPacketInfo(userId, userBaseInfoDTO, userAccountInfo);

        //填充账户信息
        fillEggAccountInfo(userId, userBaseInfoDTO, userAccountInfo);

        //填充邀请信息
        fillInviteInfo(userId, userBaseInfoDTO);

        return userBaseInfoDTO;
    }

    @Override
    public UserBriefInfoDTO getSingleUserBriefInfo(Long userId) {
        return getUserBriefInfo(userId);
    }

    @Override
    public UserDetailInfoDTO getSingleUserDetailInfo(Long userId) {
        return getUserDetailInfo(userId);
    }

    @Override
    public Map<Long, UserBriefInfoDTO> getUserBriefMapInfo(List<Long> userIdList) {
        if (CollectionUtils.isEmpty(userIdList)) {
            return Collections.emptyMap();
        }
        return getUserBriefInfoList(userIdList)
                .stream().collect(Collectors.groupingBy(UserBriefInfoDTO::getUserId,
                        Collectors.collectingAndThen(Collectors.toList(), value -> value.get(0))));
    }

    @Override
    public Map<Long, UserDetailInfoDTO> getUserDetailMapInfo(List<Long> userIdList) {
        if (CollectionUtils.isEmpty(userIdList)) {
            return Collections.emptyMap();
        }
        return getUserDetailInfoList(userIdList)
                .stream().collect(Collectors.groupingBy(UserDetailInfoDTO::getUserId,
                        Collectors.collectingAndThen(Collectors.toList(), value -> value.get(0))));
    }

    /**
     * 填充邀请信息
     *
     * @param userId          用户id
     * @param userBaseInfoDTO 用户基本信息
     */
    private void fillInviteInfo(Long userId, UserBaseInfoDTO userBaseInfoDTO) {
        EggInviteDTO eggInviteDTO = new EggInviteDTO();
        eggInviteDTO.setInviteNum(userEggInviteService.countInviteUser(userId));
        eggInviteDTO.setInviteFoodsNum(sixEnjoyInviteFoodsIntegrationService.getReceiveFoods(userId));
        eggInviteDTO.setInviteHeadImgList(userEggInviteService.getUserInviteHeadImgList(userId));

        userBaseInfoDTO.setEggInviteDTO(eggInviteDTO);
    }

    /**
     * 填充账户信息
     *
     * @param userId          用户id
     * @param userBaseInfoDTO 用户基本信息
     */
    private void fillEggAccountInfo(Long userId, UserBaseInfoDTO userBaseInfoDTO, UserAccountEntity userAccountInfo) {
        UserCouponsStatisticsDTO couponsStatistics = sixEnjoyDataStatisticsIntegrationService.getCouponsStatistics(userId);

        EggAccountDTO eggAccountDTO = new EggAccountDTO();
        eggAccountDTO.setEggCouponNum(couponsStatistics.getEggCoupon());
        eggAccountDTO.setFoodsNum(userAccountInfo.getUsableFoods());
        eggAccountDTO.setCardNum(couponsStatistics.getCardCoupon());


        userBaseInfoDTO.setEggAccountDTO(eggAccountDTO);


    }

    /**
     * 填充钱包信息
     *
     * @param userId          用户id
     * @param userBaseInfoDTO 用户基本信息
     */
    private void fillEggPacketInfo(Long userId, UserBaseInfoDTO userBaseInfoDTO, UserAccountEntity userAccountInfo) {
        EggPacketSimpleDTO eggPacketSimpleDTO = new EggPacketSimpleDTO();

        eggPacketSimpleDTO.setEnableAmount(userAccountInfo.getMoney());

        userBaseInfoDTO.setEggPacketSimpleDTO(eggPacketSimpleDTO);
    }

    /**
     * 填充联营人和代理人数据
     *
     * @param userId          用户id
     * @param userBaseInfoDTO 用户基本信息
     */
    private void fillEggAgentAndBusiness(Long userId, UserBaseInfoDTO userBaseInfoDTO) {

        EggUserRoleDTO eggUserRoleDTO = userBaseInfoDTO.getEggUserRoleDTO();

        List<Integer> roleTypeList = eggUserRoleDTO.getRoleTypeList();


        //如果是代理人则返回代理信息
        if (roleTypeList.contains(UserEggRoleEnum.COMMUNITY_AGENT.getCode())
                || roleTypeList.contains(UserEggRoleEnum.COUNTY_AGENT.getCode())
                || roleTypeList.contains(UserEggRoleEnum.CITY_AGENT.getCode())) {
            UserAssociateDataDTO userAssociateData =
                    sixEnjoyDataStatisticsIntegrationService.getUserAssociateData(userId, true);

            //返回我的联营和我的代理数据
            EggAgentInfoDTO eggAgentInfoDTO = new EggAgentInfoDTO();
            eggAgentInfoDTO.setCooperatorNum(userAssociateData.getAssociateNum());
            eggAgentInfoDTO.setDeviceNum(userAssociateData.getEquipmentNum());

            EggAgentBusinessDTO eggAgentBusinessDTO = new EggAgentBusinessDTO();
            eggAgentBusinessDTO.setDayOrderNum(userAssociateData.getOrderNumByDay());
            eggAgentBusinessDTO.setWeekOrderNum(userAssociateData.getOrderNumByWeek());
            eggAgentBusinessDTO.setMonthOrderNum(userAssociateData.getOrderNumByMonth());

            userBaseInfoDTO.setEggAgentInfoDTO(eggAgentInfoDTO);
            userBaseInfoDTO.setEggAgentBusinessDTO(eggAgentBusinessDTO);

        } else if (roleTypeList.contains(UserEggRoleEnum.BUSINESS.getCode())) {
            UserAssociateDataDTO userAssociateData =
                    sixEnjoyDataStatisticsIntegrationService.getUserAssociateData(userId,
                            false);
            EggAgentBusinessDTO eggAgentBusinessDTO = new EggAgentBusinessDTO();
            eggAgentBusinessDTO.setDayOrderNum(userAssociateData.getOrderNumByDay());
            eggAgentBusinessDTO.setWeekOrderNum(userAssociateData.getOrderNumByWeek());
            eggAgentBusinessDTO.setMonthOrderNum(userAssociateData.getOrderNumByMonth());

            userBaseInfoDTO.setEggAgentBusinessDTO(eggAgentBusinessDTO);
        }
    }

    /**
     * 填充蛋蛋佳角色信息
     *
     * @param userId          用户id
     * @param userBaseInfoDTO 用户基本信息DTO
     */
    private void fillEggRoleInfo(Long userId, UserBaseInfoDTO userBaseInfoDTO) {
        List<Integer> userRoleType = userEggRoleService.getUserRoleType(userId);

        EggUserRoleDTO eggUserRoleDTO = new EggUserRoleDTO();

        eggUserRoleDTO.setRoleTypeList(userRoleType);

        userBaseInfoDTO.setEggUserRoleDTO(eggUserRoleDTO);
    }

    /**
     * 获取用户详细信息
     *
     * @param userId 用户id
     * @return 用户详细信息
     */
    private UserDetailInfoDTO getUserDetailInfo(Long userId) {
        UserDetailInfoDTO userDetailInfoDTO = new UserDetailInfoDTO();

        UserCacheInfoBO userCacheInfo = userInfoCacheService.load(userId);
        BeanUtils.copyProperties(userCacheInfo, userDetailInfoDTO);

        //填充是否是炫彩id
        UserEquityDTO userWarmLevelDTO = userCacheInfo.getUserWarmLevelDTO();
        boolean dazzling = userWarmLevelDTO != null
                && userWarmLevelDTO.getDazzling() != null
                && userWarmLevelDTO.getDazzling();

        userDetailInfoDTO.setDazzling(dazzling);

        //设置个人简介
        UserInformationBO userInformationBO = userCacheInfo.getUserInformationBO();

        if (log.isDebugEnabled()) {
            if (userInformationBO == null) {
                log.debug("用户个人资料信息为空");
            } else {
                log.debug("用户个人资料信息：{}", JSON.toJSONString(userInformationBO));
            }
        }
        if (userInformationBO != null) {
            userDetailInfoDTO.setPersonalProfile(userInformationBO.getPersonalProfile());
        }
        //设置勋章
        userDetailInfoDTO.setWearMedalList(userCacheInfo.getWearMedalList());

        //获取勋章和权益信息
        userDetailInfoDTO.setUserWarmLevelDTO(userCacheInfo.getUserWarmLevelDTO());

        return userDetailInfoDTO;
    }


}