package com.bxm.egg.user.info.impl;

import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.bxm.egg.common.url.ProtocolFactory;
import com.bxm.egg.message.facade.service.MessageFacadeService;
import com.bxm.egg.mq.common.model.dto.PushMessage;
import com.bxm.egg.user.constant.BaseDistributeKeyConstant;
import com.bxm.egg.user.constant.UserBizConfigProperties;
import com.bxm.egg.user.convert.EggSixParamConverter;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserWechatGrantService;
import com.bxm.egg.user.info.msg.UserInfoChangeSender;
import com.bxm.egg.user.integration.EggSixEnjoyIntegrationService;
import com.bxm.egg.user.integration.MessageFacadeIntegrationService;
import com.bxm.egg.user.integration.SixEnjoyUserIntegrationService;
import com.bxm.egg.user.integration.sync.SixEnjoyUserAccountIntegrationService;
import com.bxm.egg.user.invite.egg.UserEggInviteService;
import com.bxm.egg.user.mapper.UserGrantWechatAppMapper;
import com.bxm.egg.user.model.bo.UserWechatBindBO;
import com.bxm.egg.user.model.entity.TempInviteBindEntity;
import com.bxm.egg.user.model.entity.UserGrantWechatAppEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.UserBindWxParam;
import com.bxm.egg.user.properties.SixEnjoyUrlProperties;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import com.bxm.sync.facade.dto.UserRewardDTO;
import com.bxm.sync.facade.param.UpdateUserRewardParam;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/10/8 5:50 下午
 */
@Slf4j
@Service
public class UserWechatGrantServiceImpl implements UserWechatGrantService {

    @Autowired
    private UserGrantWechatAppMapper userGrantWechatAppMapper;
    @Autowired
    private DistributedLock distributedLock;
    @Autowired
    private UserInfoService userInfoService;
    @Autowired
    private UserInfoChangeSender userInfoChangeSender;
    @Autowired
    private UserExtendService userExtendService;
    @Autowired
    private EggSixEnjoyIntegrationService eggSixEnjoyIntegrationService;
    @Autowired
    private SixEnjoyUserAccountIntegrationService enjoyUserAccountIntegrationService;
    @Autowired
    private SixEnjoyUserIntegrationService enjoyUserIntegrationService;
    @Autowired
    private UserEggInviteService userEggInviteService;
    @Autowired
    private UserBizConfigProperties userBizConfigProperties;
    @Autowired
    private MessageFacadeIntegrationService messageFacadeIntegrationService;
    @Autowired
    private SixEnjoyUrlProperties sixEnjoyUrlProperties;

    @Override
    public Optional<UserGrantWechatAppEntity> userInfoUnionIdExist(String unionId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUnionId, unionId);
        queryWrapper.eq(UserGrantWechatAppEntity::getDeleted, 0);

        UserGrantWechatAppEntity userGrantWechatAppEntity = userGrantWechatAppMapper.selectOne(queryWrapper);

        if (Objects.nonNull(userGrantWechatAppEntity)) {
            return Optional.of(userGrantWechatAppEntity);
        }

        return Optional.empty();
    }

    @Override
    public int updateWechatGrantBindUserId(Long userId, String unionId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUnionId, unionId);

        UserGrantWechatAppEntity updateEntity = new UserGrantWechatAppEntity();

        updateEntity.setUserId(userId);

        return userGrantWechatAppMapper.update(updateEntity, queryWrapper);

    }

    @Override
    public Message userBindWx(UserBindWxParam userBindWxParam) {

        String resource = buildBindWxUserLockKey(userBindWxParam.getUserId());

        if (!distributedLock.lock(resource)) {
            log.info("出现重复点击，请求参数：{}", JSON.toJSONString(userBindWxParam));
            return null;
        }

        //判断用户是否已经绑定微信
        if (userHasBindWx(userBindWxParam.getUserId())) {
            distributedLock.unlock(resource);
            return Message.build(false, "该用户已经绑定过微信，请勿重复绑定");
        }

        //如果已经存在绑定记录了则更新用户id
        if (this.userInfoUnionIdExist(userBindWxParam.getUnionId()).isPresent()) {
            //更新授权绑定信息
            updateWechatGrantBindUserId(userBindWxParam.getUserId(),
                    userBindWxParam.getUnionId());
        } else {
            UserWechatBindBO userWechatBindBO = new UserWechatBindBO();
            BeanUtils.copyProperties(userBindWxParam, userWechatBindBO);

            //添加微信授权记录
            grantBindWxRecord(userWechatBindBO);
        }

        //微信授权后更新用户头像等信息
        updateUserWxBindRelationInfo(userBindWxParam);

        //清空用户信息缓存
        userInfoChangeSender.sendUserChangeMsg(userBindWxParam.getUserId());

        //初始化资料完成度
        userExtendService.initComplete(userBindWxParam.getUserId());

        if (StringUtils.isGrateOrEqualThan(userBindWxParam.getCurVer(), "2.0.1")) {
            userBindWxHandler(userBindWxParam);
        } else {
            //调用六享微信绑定奖励
            eggSixEnjoyIntegrationService.bind(EggSixParamConverter.convert(userBindWxParam));
        }

        distributedLock.unlock(resource);
        return Message.build(true);
    }

    /**
     * 用户绑定微信后续操作
     *
     * @param userBindWxParam 用户绑定信息
     */
    private void userBindWxHandler(UserBindWxParam userBindWxParam) {

        UserRewardDTO userRewardInfo = enjoyUserIntegrationService.getUserRewardInfo(userBindWxParam.getUserId(),userBindWxParam.getDevcId());

        if (userRewardInfo == null) {
            log.error("获取用户信息失败，请检验用户在六享是否存在：{}", userBindWxParam.getUserId());
            return;
        }

        //发送用户奖励
        int sendInviteUserReward = sendUserReward(userRewardInfo, userBindWxParam);

        //说明原来就有邀请关系
        if (Objects.nonNull(userRewardInfo.getInviteUserId())) {
            //老版直接发送邀请奖励
            sendInviteUserReward(userRewardInfo.getInviteUserId(), userBindWxParam.getNickName(), sendInviteUserReward);
        } else {
            //新版本发送奖励
            newSendInviteUserReward(userBindWxParam);
        }
    }

    private void newSendInviteUserReward(UserBindWxParam userBindWxParam) {
        //邀请人逻辑
        TempInviteBindEntity userTempBindInvite = userEggInviteService.getUserTempBindInvite(userBindWxParam.getUnionId());

        if (Objects.isNull(userTempBindInvite)) {
            //用户没邀请人
            return;
        }
        UserInfoEntity userInfoEntity = userInfoService.selectAllUserById(userBindWxParam.getUserId());

        if (userInfoEntity.getCreateTime().getTime() < userTempBindInvite.getCreateTime().getTime()) {
            //如果用户创建时间小于 临时邀请时间，则不发放邀请人奖励
            //删除掉临时邀请关系
            userEggInviteService.delInviteBindByUnionId(userBindWxParam.getUnionId());
            return;
        }
        Long inviteUserId = userTempBindInvite.getUserId();

        //更新邀请人关系，更新了六享数据则会触发canal同步邀请关系，火啦不用增加邀请关系
        Boolean updateUserInvite = enjoyUserIntegrationService.updateUserInvite(userBindWxParam.getUserId(), inviteUserId);
        if (!updateUserInvite) {
            log.error("给六享绑定邀请关系失败，请检查dubbo问题还是用户已有绑定人：userId:{},inviteUserId:{}", userBindWxParam.getUserId(), inviteUserId);
            return;
        }
        //更新邀请人关系成功，则发送邀请奖励
        sendInviteUserReward(inviteUserId, userBindWxParam.getNickName(), userBizConfigProperties.getNewInviteReward());

        //删除掉临时邀请关系
        userEggInviteService.delInviteBindByUnionId(userBindWxParam.getUnionId());
    }

    private void sendInviteUserReward(Long inviteUserId, String beInviteUserName, int sendInviteUserReward) {
        if (sendInviteUserReward > 0) {
            //发送奖励
            enjoyUserAccountIntegrationService.addFoodsNum(inviteUserId,
                    sendInviteUserReward, "邀请新用户注册奖励");

            PushMessage message = PushMessage.build("邀请好友奖励已送达",
                    "您邀请的好友[" + beInviteUserName + "]已注册并绑定微信，奖励您"
                            + sendInviteUserReward + "g粮食，多邀多得哦~");
            message.assign(inviteUserId);
            message.getPayloadInfo().setProtocol(ProtocolFactory.appH5()
                    .url(sixEnjoyUrlProperties.getInvitePage())
                    .build());

            messageFacadeIntegrationService.sendPushMessage(message);
        }
    }


    private int sendUserReward(UserRewardDTO userRewardInfo, UserBindWxParam userBindWxParam) {
        int sendInviteUserReward = 0;

        UpdateUserRewardParam updateUserRewardParam = new UpdateUserRewardParam();

        if (!userRewardInfo.getRegisterReward()) {
            //发送奖励
            enjoyUserAccountIntegrationService.addFoodsNum(userBindWxParam.getUserId(),
                    userBizConfigProperties.getRegisterWxAndReward(), "新用户注册奖励");
            sendInviteUserReward += userBizConfigProperties.getInviteReward();

            updateUserRewardParam.setRegisterReward(true);
            updateUserRewardParam.setDevcId(userBindWxParam.getDevcId());
        }
        if (!userRewardInfo.getWxReward()) {
            //发送奖励
            enjoyUserAccountIntegrationService.addFoodsNum(userBindWxParam.getUserId(),
                    userBizConfigProperties.getRegisterWxAndReward(), "新用户微信绑定奖励");
            sendInviteUserReward += userBizConfigProperties.getInviteReward();

            updateUserRewardParam.setWxReward(true);
        }

        //更新信息
        updateUserRewardParam.setUserId(userBindWxParam.getUserId());
        updateUserRewardParam.setNickname(userBindWxParam.getNickName());
        updateUserRewardParam.setHeadImg(userBindWxParam.getHeadImageUrl());
        enjoyUserIntegrationService.updateUserReward(updateUserRewardParam);
        return sendInviteUserReward;
    }

    /**
     * 微信授权后更新用户头像等信息
     *
     * @param userBindWxParam 微信绑定入参
     */
    private void updateUserWxBindRelationInfo(UserBindWxParam userBindWxParam) {
        UserInfoEntity updateUserInfo = new UserInfoEntity();

        String newHeadImg = userExtendService.uploadWechatImage(userBindWxParam.getHeadImageUrl(),
                userBindWxParam,
                userBindWxParam.getUserId());

        userBindWxParam.setHeadImageUrl(newHeadImg);
        updateUserInfo.setId(userBindWxParam.getUserId());
        updateUserInfo.setHeadImg(newHeadImg);
        updateUserInfo.setNickname(userBindWxParam.getNickName());
        updateUserInfo.setSex(userBindWxParam.getSex());

        userInfoService.updateUserInfo(updateUserInfo);
    }

    /**
     * 判断用户是否已经绑定微信
     *
     * @param userId 用户id
     * @return 是否绑定
     */
    @Override
    public Boolean userHasBindWx(Long userId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUserId, userId);
        queryWrapper.eq(UserGrantWechatAppEntity::getDeleted, 0);

        List<UserGrantWechatAppEntity> grantWechatAppEntityList = userGrantWechatAppMapper.selectList(queryWrapper);

        return !CollectionUtils.isEmpty(grantWechatAppEntityList);
    }

    @Override
    public Boolean grantBindWxRecord(UserWechatBindBO userWechatBindBO) {
        // 保留微信授权信息
        UserGrantWechatAppEntity entity = new UserGrantWechatAppEntity();
        entity.setId(SequenceHolder.nextLongId());

        entity.setCountry(userWechatBindBO.getCountry());
        entity.setProvince(userWechatBindBO.getProvince());
        entity.setCity(userWechatBindBO.getCity());
        entity.setNickName(userWechatBindBO.getNickName());
        entity.setHeadImg(userWechatBindBO.getHeadImageUrl());
        entity.setOpenId(userWechatBindBO.getOpenId());
        entity.setUnionId(userWechatBindBO.getUnionId());
        entity.setSex(userWechatBindBO.getSex());

        entity.setUserId(userWechatBindBO.getUserId());
        entity.setCreateTime(new Date());

        return userGrantWechatAppMapper.insert(entity) > 0;
    }

    /**
     * 构建用户绑定微信分布式锁，保证不重复请求
     *
     * @param userId 用户id
     * @return 分布式锁key
     */
    private String buildBindWxUserLockKey(Long userId) {
        return BaseDistributeKeyConstant.USER_BIND_WX_LOCK_KEY.copy().appendKey(userId).gen();
    }
}