package com.bxm.egg.user.medal.strategy;

import com.bxm.egg.user.enums.MedalCategoryEnum;
import com.bxm.egg.user.facade.dto.UserGrantMedalDTO;
import com.bxm.egg.user.integration.DomainIntegrationService;
import com.bxm.egg.user.mapper.medal.UserMedalRelationInfoMapper;
import com.bxm.egg.user.medal.cache.AchievementMedalCacheManager;
import com.bxm.egg.user.medal.cache.UserMedalPopCacheManager;
import com.bxm.egg.user.model.dto.medal.UserMedalCounterDTO;
import com.bxm.egg.user.model.entity.medal.UserMedalInfoEntity;
import com.bxm.egg.user.model.entity.medal.UserMedalRelationInfoEntity;
import com.bxm.egg.user.properties.ClientRouteProperties;
import com.bxm.egg.user.properties.H5JumpAddressProperties;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.util.UriUtils;

import java.nio.charset.Charset;
import java.util.Date;
import java.util.List;
import java.util.Objects;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/3/6 10:46 上午
 */
@Slf4j
@Component
public abstract class AbstractMedalCounterStrategy {

    @Autowired
    private UserMedalRelationInfoMapper userMedalRelationInfoMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private UserMedalPopCacheManager userMedalPopCacheManager;

    @Autowired
    private H5JumpAddressProperties h5JumpAddressProperties;

    @Autowired
    private DomainIntegrationService domainIntegrationService;

    @Autowired
    private ClientRouteProperties clientRouteProperties;

    @Autowired
    protected AchievementMedalCacheManager achievementMedalCacheManager;

    public void handleData(UserMedalCounterDTO userMedalCounterDTO) {
        //调用模板方法
        handleCounterData(userMedalCounterDTO);
    }

    /**
     * 处理勋章统计数据
     *
     * @param userMedalCounterDTO 勋章统计信息
     */
    protected abstract void handleCounterData(UserMedalCounterDTO userMedalCounterDTO);

    /**
     * 勋章统计数据的类型
     *
     * @return 勋章统计数据的类型
     * @link com.bxm.egg.user.enums.UserMedalCounterTypeEnum
     */
    protected abstract String counterType();

    /**
     * 发放成就勋章
     *
     * @param userId              用户id
     * @param userMedalInfoEntity 成就勋章信息
     */
    protected void grantAchievementMedal(Long userId, UserMedalInfoEntity userMedalInfoEntity) {
        UserMedalRelationInfoEntity userMedalRecord = userMedalRelationInfoMapper.getByUserIdAndMedalId(userId, userMedalInfoEntity.getId());

        if (Objects.isNull(userMedalRecord)) {
            UserMedalRelationInfoEntity newGrantInfo = new UserMedalRelationInfoEntity();

            //判断是否应该佩戴更高级的勋章,为空则不需要，不为空则需要
            UserMedalRelationInfoEntity userMedalRelationInfoEntity = hasWearParentMedal(userId, userMedalInfoEntity);

            newGrantInfo.setId(sequenceCreater.nextLongId());
            newGrantInfo.setCreateTime(new Date());
            newGrantInfo.setModifyTime(new Date());
            newGrantInfo.setMedalCategory(MedalCategoryEnum.ACHIEVEMENT_MEDAL.getCode());
            newGrantInfo.setMedalLevel(userMedalInfoEntity.getLevel());
            newGrantInfo.setMedalParentId(userMedalInfoEntity.getParentId());
            newGrantInfo.setMedalId(userMedalInfoEntity.getId());
            newGrantInfo.setUserId(userId);
            if (Objects.isNull(userMedalRelationInfoEntity)) {
                newGrantInfo.setHasWear(0);
                newGrantInfo.setWearOrder(0);
            } else {
                newGrantInfo.setHasWear(1);
                newGrantInfo.setWearOrder(userMedalRelationInfoEntity.getWearOrder());
            }

            userMedalRelationInfoMapper.insertSelective(newGrantInfo);

            //将勋章信息放到用户勋章弹窗缓存中
            userMedalPopCacheManager.addUserMedalPopCache(userId, buildPopCacheData(userId, userMedalInfoEntity));

            if (log.isDebugEnabled()) {
                log.debug("为用户发放成就勋章，用户id：{}, 成就勋章id：{}", userId, userMedalInfoEntity.getId());
            }
        }
    }

    /**
     * 佩戴更高级别的勋章
     *
     * @param userId              用户id
     * @param userMedalInfoEntity 新获得的勋章
     */
    private UserMedalRelationInfoEntity hasWearParentMedal(Long userId, UserMedalInfoEntity userMedalInfoEntity) {

        List<UserMedalRelationInfoEntity> userMedalList = userMedalRelationInfoMapper.getUserMedalListByUserAndParentId(userId, userMedalInfoEntity.getParentId());

        //如果为空则返回，这是一种错误情况
        if (CollectionUtils.isEmpty(userMedalList)) {
            return null;
        }

        for (UserMedalRelationInfoEntity userMedalRelationInfoEntity : userMedalList) {
            if (userMedalRelationInfoEntity.getHasWear() == 1) {
                //取消原来等级勋章的佩戴
                UserMedalRelationInfoEntity minLevelMedalRelationInfo = new UserMedalRelationInfoEntity();
                minLevelMedalRelationInfo.setId(userMedalRelationInfoEntity.getId());
                minLevelMedalRelationInfo.setModifyTime(new Date());
                minLevelMedalRelationInfo.setWearOrder(0);
                minLevelMedalRelationInfo.setHasWear(0);

                userMedalRelationInfoMapper.updateByPrimaryKeySelective(minLevelMedalRelationInfo);

                return userMedalRelationInfoEntity;
            }
        }

        return null;
    }

    /**
     * 构建弹窗缓存数据对象
     *
     * @param userId              用户id
     * @param userMedalInfoEntity 源数据
     * @return 目标对象
     */
    private UserGrantMedalDTO buildPopCacheData(Long userId, UserMedalInfoEntity userMedalInfoEntity) {
        UserGrantMedalDTO userGrantMedalDTO = new UserGrantMedalDTO();

        userGrantMedalDTO.setMedalId(userMedalInfoEntity.getId());
        userGrantMedalDTO.setGrantDate(new Date());
        userGrantMedalDTO.setMedalImgUrl(userMedalInfoEntity.getGifImgUrl());
        userGrantMedalDTO.setMedalName(userMedalInfoEntity.getName());
        userGrantMedalDTO.setJumpUrl(buildMyMedalPageUrl(userId));

        return userGrantMedalDTO;
    }

    /**
     * 构建跳转我的勋章页面url
     *
     * @param userId 用户id
     * @return 页面地址url
     */
    private String buildMyMedalPageUrl(Long userId) {
        String url = domainIntegrationService.getInnerH5BaseUrl() +
                String.format(h5JumpAddressProperties.getMyMedalUrl(), userId);

        url = UriUtils.encode(url, Charset.defaultCharset());

        return String.format(clientRouteProperties.getWebView(), url, UriUtils.encode("我的勋章", Charset.defaultCharset()));
    }
}