package com.bxm.localnews.activity.controller;

import java.util.List;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;

import com.bxm.localnews.activity.common.constant.PrivilegeStatusEnum;
import com.bxm.localnews.activity.dto.*;
import com.bxm.localnews.activity.param.*;
import com.bxm.localnews.activity.service.MerchantService;
import com.bxm.localnews.activity.service.PrivilegeParticipantService;
import com.bxm.localnews.activity.service.PrivilegeService;
import com.bxm.localnews.common.annotation.TouTiaoAuth;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.message.constant.SmsTemplateEnum;
import com.bxm.localnews.message.service.SmsSupplyService;
import com.bxm.localnews.user.service.UserService;
import com.bxm.localnews.user.vo.User;
import com.bxm.newidea.component.util.WebUtils;
import com.bxm.newidea.component.vo.PageWarper;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.regexp.RE;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

/**
 * Created by Arei on 2018/10/17.
 */
@Api(tags = "2-06 优惠活动相关接口", description = "包括活动的相关接口操作")
@RestController
@RequestMapping("/api/privileges")
public class PrivilegeController {
    @Resource
    private PrivilegeService privilegeService;
    @Resource
    private PrivilegeParticipantService privilegeParticipantService;
    @Resource
    private MerchantService merchantService;
    @Resource
    private SmsSupplyService smsSupplyService;
    @Resource
    private UserService userService;

    @ApiOperation(value = "2-06-1 获得优惠活动列表", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "longitude", value = "经度", required = false),
            @ApiImplicitParam(name = "latitude", value = "纬度", required = false),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true, dataType = "long"),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true)
    })
    @TouTiaoAuth
    @GetMapping
    public Json<PageWarper<PrivilegeListDTO>> listPrivileges(@Validated PrivilegeParam privilegeParam) {
        return Json.build(privilegeService.listPrivilege(privilegeParam));
    }

    @ApiOperation(value = "2-06-2 获得优惠活动详情", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "活动id", required = true, dataType = "long"),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true, dataType = "long"),
    })
    @TouTiaoAuth
    @GetMapping("/detail")
    public Json<PrivilegeDTO> getPrivilegeById(@RequestParam("id") Long id,
                                               @RequestParam("userId") Long userId) {
        PrivilegeDTO privilegeDTO = privilegeService.getPrivilegeDetail(id, userId);
        if (privilegeDTO == null)
            return Json.build(RespCode.BAD_REQUEST, "活动不存在");
        return Json.build(privilegeDTO);
    }


    @ApiOperation(value = "2-06-3 获得活动抽奖结果名单", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "privilegeId", value = "活动id", dataType = "long", required = true)
    })
    @TouTiaoAuth
    @GetMapping("/result/{privilegeId}")
    public Json<List<WinnerDTO>> getPrivilegeWinnersById(@PathVariable("privilegeId") Long privilegeId) {
        return Json.build(privilegeService.listWinnerByPrivilegeId(privilegeId));
    }

    @ApiOperation(value = "2-06-4 参加活动", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "privilegeId", value = "活动id", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true),
            @ApiImplicitParam(name = "inviteUserId", value = "邀请人用户ID", required = false),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true)
    })
    @TouTiaoAuth
    @PostMapping("/participate")
    public Json<ParticipateResultDTO> participate(@Validated ParticipatePrivilegeParam participatePrivilegeParam,
                                                  HttpServletRequest httpServletRequest) {
        PrivilegeDTO privilegeDTO = privilegeService.getPrivilegeDetail(participatePrivilegeParam.getPrivilegeId(), participatePrivilegeParam.getUserId());
        if (privilegeDTO == null) {
            return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("活动不存在");
        }
        PrivilegeStatusEnum statusEnum = PrivilegeStatusEnum.getEnumByStatus(privilegeDTO.getStatus());
        switch (statusEnum) {
            case END:
                return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("活动已结束");
            case CANCEL:
                return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("活动已取消");
            case WAIT_TO_START:
                return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("活动未开始");
            case PROCESSING:
                break;
            default:
                return Json.build(new ParticipateResultDTO(3)).setCode(RespCode.INTERNAL_SERVER_ERROR).setErrorMsg("活动错误");

        }
        if (privilegeDTO.getParticipated()) {
            return Json.build(new ParticipateResultDTO(1)).setCode(RespCode.BAD_REQUEST).setErrorMsg("已经参加过");
        }
        if (participatePrivilegeParam.getPlatform() != 3) {
            if (!privilegeDTO.getAreaCodeList().contains(participatePrivilegeParam.getAreaCode())) {
                return Json.build(new ParticipateResultDTO(2)).setCode(RespCode.BAD_REQUEST).setErrorMsg("不能跨地域参加活动");
            }
        }
        String ip = WebUtils.getIpAddr(httpServletRequest);
        int result = privilegeParticipantService.participantPrivilege(
                participatePrivilegeParam.getUserId(),
                participatePrivilegeParam.getInviteUserId(),
                participatePrivilegeParam.getPrivilegeId(),
                ip
        );
        if (result == 1) {
            return Json.build(new ParticipateResultDTO(0));

        } else {
            return Json.build(new ParticipateResultDTO(3)).setCode(RespCode.INTERNAL_SERVER_ERROR).setErrorMsg("发生错误");
        }
    }

    @ApiOperation(value = "2-06-5 当前用户邀请的好友列表", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "privilegeId", value = "活动id", dataType = "long", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long", required = true)
    })
    @TouTiaoAuth
    @GetMapping("/invitees")
    public Json<PageWarper<PrivilegeParticipantDTO>> getInviteesByUserId(@Validated PrivilegeParticipantParam privilegeParticipantParam) {
        PageWarper<PrivilegeParticipantDTO> privilegeParticipantDTOPageWarper
                = privilegeService.listPrivilegeParticipantByInvite(privilegeParticipantParam);
        return Json.build(privilegeParticipantDTOPageWarper);
    }

    @ApiOperation(value = "2-06-6 获得推荐活动列表", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "longitude", value = "经度", required = false),
            @ApiImplicitParam(name = "latitude", value = "纬度", required = false),
            @ApiImplicitParam(name = "privilegeId", value = "当前活动ID", required = false, dataType = "long"),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = false, dataType = "long"),
            @ApiImplicitParam(name = "areaCode", value = "地区编码", required = true)
    })
    @TouTiaoAuth
    @GetMapping("/recommend")
    public Json<PageWarper<PrivilegeListDTO>> listRecommendPrivileges(PrivilegeRecommendParam privilegeRecommendParam) {
        return Json.build(privilegeService.listRecommendPrivilege(privilegeRecommendParam));
    }

    @ApiOperation(value = "2-06-7 商家报名", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "name", value = "商家名字（公司名）", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", required = true, dataType = "long"),
            @ApiImplicitParam(name = "contractor", value = "联系人姓名", required = true),
            @ApiImplicitParam(name = "code", value = "短信验证码", required = true),
            @ApiImplicitParam(name = "phone", value = "联系电话", required = true)
    })
    @TouTiaoAuth
    @PostMapping("/merchant")
    public Json newMerchant(@Validated MerchantParam merchantParam) {
        User user = userService.getPhone(merchantParam.getUserId());
        boolean result = smsSupplyService.verifySmsCode(user.getPhone(), merchantParam.getCode(), SmsTemplateEnum.MERCHANT);
        if (!result) {
            return Json.build(RespCode.BAD_REQUEST, "验证码错误");
        }
        int insertResult = merchantService.saveMerchant(merchantParam);
        if (insertResult == 1) {
            return Json.build("成功");
        } else {
            return Json.build(RespCode.INTERNAL_SERVER_ERROR, "失败");
        }
    }
}
