package com.bxm.localnews.activity.controller;

import java.util.List;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.activity.dto.*;
import com.bxm.localnews.activity.param.RecordWechatParam;
import com.bxm.localnews.activity.service.VipService;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.dto.UserInfoDTO;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RestController;

/**
 * @author zhaoyadong 2018/12/24 10:30
 * @desc
 */
@Api(tags = "2-07 VIP相关接口", description = "包括VIP的相关接口操作")
@RestController
@RequestMapping("/api/vip")
public class VipController {

    private static final Logger logger = LoggerFactory.getLogger(VipController.class);

    private VipService vipService;

    private UserIntegrationService userIntegrationService;

    @Autowired
    public VipController(VipService vipService, UserIntegrationService userIntegrationService) {
        this.vipService = vipService;
        this.userIntegrationService = userIntegrationService;
    }

    @ApiOperation(value = "2-07-1 获取成功邀请好友的列表", notes = "在首页引导流程和用户激活vip进度时调用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long", required = true)
    })
    @GetMapping("/invite/success")
    public Json<UserSuccessFriendDTO> inviteFriendSuccess(Long userId) {
        //todo 当用户未绑定微信时，抛出401让用户重新绑定微信
        UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(userId);
        logger.info("vip邀请好友时用户信息:{}", JSON.toJSON(userInfoDTO));
        if (StringUtils.isEmpty(userInfoDTO.getWeixin())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "您未绑定微信,请退出账号后重新登录");
        }
        return ResultUtil.genSuccessResult(vipService.getSuccessFriend(userId));
    }

    @ApiOperation(value = "2-07-2 查看用户分享的vip页面的用户微信好友列表", notes = "用于查看用户微信好友查看用户分享VIP页面")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long", required = true)
    })
    @GetMapping("/invite/record")
    public Json<List<WechatUserInfoDTO>> inviteFriendRecord(Long userId) {
        return ResultUtil.genSuccessResult(vipService.getInviteFriendRecord(userId));
    }

    @ApiOperation(value = "2-07-3 记录用户查看记录", notes = "记录用户微信好友查看用户分享vip的页面信息")
    @PostMapping("/record")
    public Json recordUser(RecordWechatParam recordWechatParam) {
        vipService.addViewRecord(recordWechatParam);

        return ResultUtil.genSuccessMsg("添加成功");
    }

    @ApiOperation(value = "2-07-4 获取客服信息", notes = "返回客服二维码地址和微信号")
    @GetMapping("/customer/service")
    public Json<CustomerServiceDTO> customerService() {
        return ResultUtil.genSuccessResult(vipService.getCustomerService());
    }

    @ApiOperation(value = "2-07-5 预领取vip", notes = "记录用户领取vip卡但是不激活，领取vip卡时调用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "areaCode", value = "地区编码", dataType = "String", required = true),
            @ApiImplicitParam(name = "unionId", value = "用户微信ID", dataType = "String", required = true),
            @ApiImplicitParam(name = "wechatName", value = "用户微信昵称", dataType = "String", required = true),
            @ApiImplicitParam(name = "wechatImg", value = "用户微信头像", dataType = "String", required = true)
    })
    @PostMapping("/receive")
    public Json<WechatVipInfoDTO> advanceReceiveVip(String unionId, String areaCode, String wechatName, String wechatImg) {
        return ResultUtil.genSuccessResult(vipService.addWechatUserVip(areaCode, unionId, wechatName, wechatImg));
    }

    @ApiOperation(value = "2-07-6 更新用户VIP引导标记", notes = "引导用户查看vip获取流程后更新用户不再显示引导页时调用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long", required = true)
    })
    @PostMapping("/guide")
    public Json updateGuide(Long userId) {

        return ResultUtil.genSuccessResult(vipService.modifyUserGuid(userId));
    }

    @ApiOperation(value = "2-07-7 通过用户id查看vip信息", notes = "用户进入app内获取对应vip信息时调用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "areaCode", value = "地区编码", dataType = "String", required = true),
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long", required = true)
    })
    @GetMapping("/view/userId")
    public Json<UserVipInfoDTO> viewUserVip(String areaCode, Long userId) {
//        //todo 当用户未绑定微信时，抛出401让用户重新绑定微信
//        UserInfoDTO userInfoDTO = userService.getUserFromRedisDB(userId);
//        logger.error("vip邀请好友时用户信息:{}", JSON.toJSON(userInfoDTO));
//        if (StringUtils.isEmpty(userInfoDTO.getWeixin())) {
//            return ResultUtil.genFailedResult(RespCode.UNAUTHORIZED, "token error");
//        }
        return ResultUtil.genSuccessResult(vipService.modifyAndGetUserVip(userId, areaCode));
    }

    @ApiOperation(value = "2-07-8 通过微信uid查看VIP信息", notes = "微信授权登陆时获取用户vip信息，h5微信端调用")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "areaCode", value = "地区编码", dataType = "String", required = true),
            @ApiImplicitParam(name = "unionId", value = "用户微信ID", dataType = "String", required = true)
    })
    @GetMapping("/view/unionId")
    public Json<WechatVipInfoDTO> viewUserVip(String unionId, String areaCode) {
        return ResultUtil.genSuccessResult(vipService.getWechatVip(unionId, areaCode));
    }

    @ApiOperation(value = "2-07-9 查看vip已激活的轮播信息", notes = "只返回用户昵称列表，用于查看用户vip分享页面时调用")
    @GetMapping("/view/rank")
    public Json<List<WechatUserInfoDTO>> viewVipSuccessRank() {
        return ResultUtil.genSuccessResult(vipService.getVipRank());
    }

    @ApiOperation(value = "2-07-10 更新用户不再提示显示浮层", notes = "用于app首页是否显示用户激活vip卡显示的浮层")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long", required = true)
    })
    @PostMapping("/view/layer")
    public Json<Message> viewUserLayer(Long userId) {
        return ResultUtil.genSuccessResult(vipService.modifyUserShowLayer(userId));
    }

    @ApiOperation(value = "2-07-11 获取用户邀请成功的所有好友列表", notes = "用于显示用户邀请好友列表")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户ID", dataType = "long", required = true)
    })
    @GetMapping("/invite/success/all")
    public Json<List<WechatUserInfoDTO>> getWechatUserList(Long userId) {
        return ResultUtil.genSuccessResult(this.vipService.getAllSuccessWechatUserList(userId));
    }
}